# Rendering forms and Validating form data

We have a `utilities.forms.core` library which simplifies forms. It has separate utility functions for:

- defining form fields
- for parsing form data
- for validating data
- for rendering errors and fields.

Each of it is handled and defined separately.

A `form-field` function takes a name and optional attributes of a form field responsible for html rendering. It provides a map of `label`, `name`, html `attrs`, `:optional?` indicator, `help-text` and related values in case of `select` or `radio` fields.

The `checks` special-form runs all the validations and provides a map of errors.

The utility functions `get-post-data` and `get-fields-data` return a keyword map from given request or data for the fields provided.

We will have a detailed look at each of them below.


## Defining Form Fields

Form fields are mostly used for rendering. We can generate the `field` map using the `form-field` function. By default, it will make the form-field non-blank unless we set the `:optional?` keyword to `true` in options.


```clojure
(require '[utilities.forms.core as f]
         '[utilities.forms.checks :as v])

; Email field
(f/form-field :email {:help-msg "We promise we won't spam"})

; Password field
(f/form-field :password {:attrs  {:type "password"}})

; Optional phone-number field
(f/form-field :phone-number {:optional? true})

; Textarea field for long message
(f/form-field :message {:field-type f/textarea-field
                        :attrs {:cols 60 :rows 5}})

; Select box to select a plan from various plans
(f/form-field :plan {:field-type f/select-field
                     :options    {"hobby"    "Hobby (limited usage)"
                                  "personal" "Unlimited for 1 Person"
                                  "family"   "Unlimited for 4"}})

; Checkbox for agreeing terms and conditions
(f/form-field :agree-terms? {:field-type f/checkbox-field})


; Optional Checkbox for receiving updates
(f/form-field :get-updates? {:field-type f/checkbox-field
                             :optional? true})
```


## Defining Validations

For example, we want to create a user registration form. We would need 3 fields: `email`, `password` and `confirm-password`. The `email` field should check for valid email address and the `password` and `confirm-password` field should be over 6 characters. At total form level, we would also need to check that `password` is equal to `confirm-password`.

Defining fields and validations in `rules.clj`

```clojure
(require '[utilities.forms.core :as f]
         '[utilities.forms.checks :as v])

; Defining checks and fields in rules.clj
(def email (f/form-field :email))

(def password (f/form-field :password {:attrs {:type "password"}}))

(def confirm-password (f/form-field :confirm-password
                                    {:attrs {:type "password"}}))

(def registration-fields [email password confirm-password])

(def get-registration-errors
  [{:keys [email password confirm-password] :as data}]
  (f/checks
    (f/check-non-blank registration-fields data)
    :email              (v/email? email)
    :email              (data/unregistered? email)
    :password           (v/strong-password? password)
    :confirm-password   (v/matching-passwords? password confirm-password)))
```


## Validating a Form

Using validations in `views.clj`

`todo`


## Rendering a form

```clojure
(f/render-form rules/registration-fields form-data errors)
; will render the html with all the fields with their :data values and their error-messages if any
```
