(ns manager
  (:require [clojure.string :as s]
            [clojure.pprint :as pprint]
            [clojure.java.shell :as shell]
            [utilities.files :as f]
            [inflections.core :as i]
            [clojure.java.io :as io]
            [clj-jgit.porcelain :as g]))


(def resources (list "src/utilities" "doc" "test/utilities"))

(def resource-edn (io/resource "managed-sources.edn"))

(def template-path "resources/leiningen/new/jurassic_web")


(defn update-resources-edn
  "Updates managed-sources.edn with list of resource files"
  [resources]
  (let [paths   (concat (for [r resources
                              f (f/list-files (f/path template-path r))
                              :when (not (f/matches? f ".DS_Store"))]
                          (f/path r f)))]
    (with-open [w (io/writer resource-edn)]
      (pprint/pprint paths w))))


(defn missing-in-source?
  "Returns true if not destination file is missing in source"
  [src-path dest-path]
  (let [sources       (set (f/list-files src-path))
        destinations  (set (f/list-files dest-path))
        diff          (clojure.set/difference destinations sources)]
    (when (seq diff)
      (throw (Exception.
               (str diff " files in destination but not in source"))))))


(defn hard-link-file
  "Creates hard link for src file to given destination"
  [src-file dest-file]
  (println (shell/sh "ln" "-vf" src-file dest-file)))


(defn hard-link-dir
  "Creates hard links for all files in given directory"
  [src-dir dest-dir]
  (when-not (missing-in-source? src-dir dest-dir)
    (seq (for [f (f/list-files src-dir)]
           (hard-link-file (f/path src-dir f)
                           (f/path dest-dir f))))))


(def not-empty? (complement empty?))

(defn- sh
  "Calls the shell command and throws errors if any is raised.
  Takes an :allowed? function to allow failures.
  "
  [& args]
  (let [[args {:keys [allowed?]}] (split-with (complement keyword?) args)
        _                         (println "Calling " args)
        res                       (apply shell/sh args)
        error?                    (or (not-empty? (:err res))
                                      (not= (:exit res) 0))]
    (println res)

    (if (and error? (not (allowed? res)))
      (throw (Exception. ^String (str "Exited with " (:exit res)
                                      "\nERROR:" (:err res)))))

    res))


(defn git-branch
  []
  (let [out         (:out (sh "git" "branch"))
        names-*     (map s/trim (s/split-lines out))
        current-*   (first (filter #(s/starts-with? % "* ") names-*))
        names       (remove #(= current-* %) names-*)
        current     (s/replace-first current-* "* " "")
        branches    (cons current names)]
    (println "BRANCHES" branches)
    branches))


(defn git-create-or-checkout
  "Create a git branch with given name"
  [repo project-name]
  (if-not (contains? (set (git-branch)) project-name)
    (sh "git" "branch" project-name))
  (g/git-checkout repo project-name))


(defn pull-from-project
  "Creates a git branch to track project-path"
  [project-path project-name]
  (println "Project path" project-path)
  (let [repo          (g/load-repo ".")]
    (g/git-checkout repo "master")

    ; switch to project branch
    (git-create-or-checkout repo project-name)
    
    ; copy files from project
    (doseq [r resources]
      (f/copy-dir (f/path project-path r) (f/path template-path r)))

    ; do a commit
    (sh "git" "add" "." "-A")
    (sh "git" "commit" "-m" (str "Added updates from " project-path)
        :allowed? (fn [res] (s/includes? (:out res) "nothing to commit")))

    ; merge master
    (sh "git" "merge" "master")

    ; reverse merge in master
    (g/git-checkout repo "master")
    (sh "git" "merge" project-name)

    ; update sources in edn
    (update-resources-edn resources)

    (println "WOOHOO!!! Ready for release. Update version.")
    (println "export GPG_TTY=$(tty)")
    (println "lein deploy clojars")
    (println "EyD")))


(defn push-to-project
  "Pushes latest jurassic template to destination project"
  [project-path project-name]
  (let [repo  (g/load-repo ".")]
    (git-create-or-checkout repo project-name)
    (sh "git" "merge" "master")

    ; stop if differences from master as it needs to incorporated in this template
    (if-not (empty? (:out (sh "git" "diff" "master")))
      (throw (Exception. ^String (str "The project branch differs from master.\n"
                                      "INCORPORATE PROJECT BRANCH IN MASTER"))))

    (sh "lein" "new" "jurassic-web" project-name
        "--to-dir" project-path "--force")

    (g/git-checkout repo "master")))
