(ns utilities.db
  (:require [clojure.java.jdbc :as jdbc]
            [utilities.settings :as settings]
            [clojure.string :as s]
            [utilities.templates :as t]
            [utilities.exceptions :as ex])
  (:import (com.zaxxer.hikari HikariDataSource)))


; Function to BUILD a SELECT query
(defn- to-coll
  "Converts the string to coll with string.
  Returns same if already a collection"
  [x]
  (when x
    (if (coll? x)
      x
      [x])))


(defn- integer
  "Converts given value to number else nil"
  [x]
  (when x
    (Integer. x)))


(defn query
  "Create a SQL query map"
  ([from parts] (query from parts {}))
  ([from {:keys [select where group-by having order-by limit offset]} params]
   (let []
     {:select    (concat (:select from) (to-coll select))
      :from      (get from :from from)
      :where     (concat (:where from) (to-coll where))
      :group-by  (or group-by (:group-by from))
      :having    (or having (:having from))
      :order-by  (concat (:order-by from) (to-coll order-by))
      :limit     (or (integer limit) (:limit from))
      :offset    (or (integer offset) (:from offset))
      :params    (merge (:params from) params)})))


(defn- order-params
  "Orders map of params in the order of placeholders"
  [params placeholders]
  (reduce (fn [coll key]
            (let [val   (get params key)
                  val   (if (= val nil)
                          ""
                          val)]
              (conj coll val)))
          []
          placeholders))


(defn query-params
  "Converts query string with placeholders into SQL query with ? holders"
  [q-str params]
  (let [parts     (t/break-text q-str)
        holders   (filter keyword? parts)
        params    (order-params params holders)
        q-str     (s/join "" (for [part parts]
                               (if (keyword? part) "?" part)))]
    (cons q-str params)))


(defn sql
  "Converts query map to SQL string"
  [q]
  (let [select    (str "SELECT "
                       (let [clauses (seq (:select q))]
                         (if clauses
                           (s/join ", " clauses)
                           "*")))
        from      (str "FROM " (:from q))
        where     (when-let [clauses (seq (:where q))]
                    (str "WHERE " (s/join " AND " clauses)))
        group-by  (when-let [clause (:group-by q)]
                    (str "GROUP BY " clause))
        having    (when-let [clause (:having q)]
                    (str "HAVING " clause))
        order-by  (when-let [clauses (seq (:order-by q))]
                    (str "ORDER BY " (s/join ", " clauses)))
        limit     (when-let [n (:limit q)]
                    (str "LIMIT " n))
        offset    (when-let [n (:offset q)]
                    (str "OFFSET " n))
        q-str     (s/join "\n" (remove nil?
                                       [select from where group-by
                                        having order-by limit offset]))]
    (query-params q-str (:params q))))


; Connection Pooling
; We are using dynamic connection to enable transactions when needed

(def ^:dynamic connection (settings/db-spec))

(defn get-pooled-connection
  "Creates Database connection pool to be used in queries"
  []
  (let [{:keys [subname user password]} (settings/db-spec)
        pool (doto (HikariDataSource.)
               (.setJdbcUrl (str "jdbc:mysql:" subname))
               (.setUsername user)
               (.setPassword password))]
    {:datasource pool}))

(defn enable-pooling
  "Sets connection to use pooled connection"
  []
  (def ^:dynamic connection (get-pooled-connection)))


; Functions to EXECUTE the query
; Documentation for jdbc api: https://clojure.github.io/java.jdbc/#clojure.java.jdbc/execute!

(defn get!
  "Returns first result for the given query.
  Raises error if none or more than one result exist."
  [q]
  (let [result    (jdbc/query connection (sql q))
        n         (count result)]
    (cond
      (= n 1) (first result)
      (> n 1) (ex/raise ::db-error "Found Multiple rows")
      (= n 0) (ex/raise ::db-error "No results found"))))


(defn get-or-404
  "Raises page-not-found error if get! fails"
  [q]
  (try
    (get! q)
    (catch Exception e
      (if (ex/cause? e ::db-error)
        (ex/raise ::ex/page-not-found "Resource not found")
        (throw e)))))


(defn first!
  "Limits and returns first record for given query
  Can return nil"
  [q]
  (let [q     (assoc q :limit 1)]
    (first (jdbc/query connection (sql q)))))


(defn count!
  "Runs MySQL count query for given query"
  [q]
  (let [q     (assoc q :select ["COUNT(*) as __count"])
        res   (jdbc/query connection (sql q))]
    (:__count (first res))))


(defn all!
  "Returns all results for the query"
  [q]
  (jdbc/query connection (sql q)))


(defn create
  "Create new table row"
  [table row]
  (try
    (let [res (jdbc/insert! connection table row)]
      (get! (query table
                   {:where "id = ::id"}
                   {:id (:generated_key (first res))})))
    (catch java.sql.SQLIntegrityConstraintViolationException e
      (if (s/includes? (ex/message e) "Duplicate entry")
        (ex/raise ::db-duplicate-error "Record already exists")
        (throw e)))))


(defn update!
  "Update table row where the clause matches"
  ([table set-map where]
   (jdbc/update! connection table set-map where))

  ([q set-map]
   (let [table (:from q)
         where (query-params (s/join " AND " (:where q)) (:params q))]
     (jdbc/update! connection table set-map where))))


(defn insert-or-update
  "Performs insert-on-duplicate-update query"
  ([table data update-columns]
   (let [column-names  (keys (first data))
         values        (map vals data)]
     (insert-or-update table column-names values update-columns)))

  ([table column-names values update-columns]
   (let [update-columns     (map name update-columns)
         column-names       (map name column-names)
         update-columns-ref (s/join ", " (for [col update-columns]
                                           (str col " = VALUES(" col ")")))
         q-str (str "INSERT INTO "
                    table " ("
                    (s/join ", " column-names)
                    ") VALUES ("
                    (s/join ", " (repeat (count column-names) "?"))
                    ") ON DUPLICATE KEY UPDATE "
                    update-columns-ref)
         query (cons q-str values)]
     (try
       (jdbc/execute! connection query {:multi? true})
       (catch Exception e
         (println "QUERY TRIED: " query)
         (throw e))))))


(defn delete!
  "Delete table row where the clause matches"
  ([table where] (jdbc/delete! connection table where))

  ([q]
   (let [table (:from q)
         where (query-params (s/join " AND " (:where q))
                             (:params q))]
     (jdbc/delete! connection table where))))


(defmacro using-transaction
  "Runs given commands in a new transaction connection"
  [& body]
  `(jdbc/with-db-transaction [t-conn# (settings/db-spec)]
     (binding [connection t-conn#]
       ~@body)))
