(ns utilities.forms.core
  (:require [hiccup.core :refer [html h]]
            [clojure.string :as s]
            [utilities.forms.components :as field]))

; Field types

(def input-field ::field/input)
(def custom-field ::field/custom-field)
(def textarea-field ::field/textarea)
(def checkbox-field ::field/checkbox)
(def hidden-field ::field/hidden)
(def select-field ::field/select)


(defn form-field
  "Generates a field map with proper defaults"
  ([field-name] (form-field field-name {}))

  ([field-name {:keys [field-type label attrs help-msg optional?
                       renderer options]}]

   (when (and (= field-type custom-field)
              (not renderer))
     (throw (Exception.
              (str "Renderer not provided for custom field " field-name))))

   (when (and renderer (not= field-type custom-field))
     (throw (Exception. "Renderer is supported only for custom field")))

   (when (and (= field-type select-field)
              (not (map? options)))
     (throw (Exception.
              (str "No options provided for select field " field-name))))

   {:field-type (or field-type input-field)
    :name       (keyword field-name)
    :label      (or label (s/capitalize (name field-name)))
    :attrs      attrs
    :help-msg   help-msg
    :optional?  optional?
    :renderer   renderer
    :options    options}))


(defn render-form
  "renders form with errors and data"
  [fields data errors]
  (let [fields-html (for [field fields]
                      (let [key (:name field)]
                        (field/render field (key data) (key errors))))
        form-msg (when-let [msg (::form errors)]
                   (html [:div.alert.error (h msg)]))]
    (str form-msg (s/join "\n" fields-html))))


(defn get-fields-data
  "Returns map of field names and corresponding data from data map"
  [fields data]
  (into {} (for [field fields]
             (let [key (:name field)]
               [key (get data key
                         (get data (name key)))]))))


(defn get-post-data
  "Parses post data from request and returns matching map based on fields
  If initial data is provided, then post data is merged into it."
  ([fields request]
   (let [post-data (:form-params request)]
     (if-not (empty? post-data)
       (get-fields-data fields post-data)))))


; Form validation is responsible for checking
; - optional fields
; - error messages
; - mapping messages to fields

(defn validate-not-blank
  "Returns true if the stripped value is a string of over 1 character"
  [value]
  (if (and (string? value)
           (not (empty? (s/trim value)))
           (not (nil? value)))
    ::is-clean
    "This is a required field"))


(defn get-error
  "Returns map of key and result if the result is not clean"
  [key result]
  (when-not (= result ::is-clean)
    {key result}))


(defn get-non-blank-errors
  "Returns map of errors for blank non-optional fields"
  [fields data]
  (let [fields  (remove :optional? fields)
        errors  (reduce (fn [errors field]
                          (let [name (:name field)
                                value (get data name)]
                            (merge errors (get-error name (validate-not-blank value)))))
                        {}
                        fields)]
    (when-not (empty? errors)
      errors)))
