(ns utilities.forms.core-test
  (:require [clojure.test :refer :all]
            [utilities.forms.components :as c]
            [utilities.exceptions :as ex]
            [utilities.forms.validations :as v]
            [utilities.forms.core :as f]
            [ring.mock.request :as mock]
            [clojure.string :as s]))

; Test Form

(defn validate-unique-username
  [username]
  (if-not (= (s/lower-case username) "foo")
    ::f/is-clean
    "Username not available"))

(def username-field (f/form-field :username))
(def email-field (f/form-field :email {:optional? true}))
(def password-field (f/form-field :password))

(def registration-fields [username-field email-field password-field])

(defn get-registration-errors
  [data]
  (let [{:keys [username email password]} data
        errors (f/get-non-blank-errors registration-fields data)
        errors (merge errors
                      (when-not (empty? email)
                        (f/get-error :email (v/validate-email email)))
                      (f/get-error :password (v/validate-strong-password password)))]
    (or errors
        (f/get-error ::f/form (validate-unique-username username)))))


; Tests

(deftest test-registration-form

  (testing "error messages"
    (are [data errors]
      (= (get-registration-errors data) errors)

      {:username "bar"
       :email ""
       :password "foo@bar"}
      nil

      {:username "foo"
       :email "foo@bar.com"
       :password "foo@bar"}
      {::f/form "Username not available"}

      {:username "foo"
       :email "foo@bar"
       :password "fo"}
      {:email "Please provide a valid email"
       :password "Password should be minimum 6 characters"}

      {:username ""}
      {:username "This is a required field"
       :password "Password should be minimum 6 characters"}))

  (testing "post data"
    (are [req data]
      (= (f/get-post-data registration-fields {:form-params req})
         data)

      {"username" "foo"
       "overload" "bar"}
      {:username "foo" :email nil :password nil})))


(deftest test-fill-form


  (testing "fill-form"
    (let [person  {:id 1, :username "Harry", :password "bar"}
          data    (f/get-fields-data registration-fields person)]
      (is (= data {:username "Harry" :email nil :password "bar"}))))

  (testing "incremental fill"
    (let [default   {:username "Harry", :email nil, :password "bar"}
          i         {:password "baz"}
          data      (f/get-fields-data registration-fields default)
          data      (merge data i)]
      (is (= data {:username "Harry" :email nil :password "baz"}))))

  (testing "bad incremental fill"
    (let [default   {:id 1, :username "Harry", :email "foo@bar.com", :password "bar"}
          i         {:email "foo@bar.com" :password "baz", :id 3}
          data      (merge default (f/get-fields-data registration-fields i))]
      ; the username should be nil as it is a form field but not provided in submit
      ; the id is retained as 1 as it is not a form field
      (is (= data {:id 1 :username nil :email "foo@bar.com" :password "baz"})))))


(deftest test-render-fields
  (testing "render textarea"
    (let [field (f/form-field "message" {:field-type ::c/textarea})]
      (is (= (str "<label for=\"id_message\">Message</label>"
                  "<textarea class=\"u-full-width\" id=\"id_message\" name=\"message\"></textarea>")
             (c/render field nil nil))))))
