# Project Structuring

We should break our project into lots of sub-pieces called apps. A project is a website as a whole. A library website can be broken into apps such as `users`, `books` and `memberships`. Similarly a blogging website can have apps like `posts`, `comments` and `authors`.

The general files which belong to project as a whole should go into `project-name` directory. Example the base template of the website provides a common look and feel to all the apps.

**Project Structure**

```
project-dir
    doc: for project documentation
    resources: for all the non clojure files
    src: for the clojure source code (namespaces)
    test: for all the tests


src
    - utilities
    - project_name
        - handler.clj: for all the website routes
        - pages.clj: for all the base templates
    - app_1_name
    - app_2_name
```

## Utilities

Utilities are like small libraries which are not dependent on the project code. The should be such that we can copy the utilities folder in any new project without any modification.


## Application Specific Files

`data.clj`: is the top most low level abstraction mostly consumed by `rules.clj` and `views.clj`.

`rules.clj`: is the layer for handling logic related abstractions of the project. It describes the validations, user-permissions and provides processing queues. Apps communicate among them through `rules.clj`.



**Say we are creating an application to get all Company Announcements.** We can structure it as below.

We would need parsers and fetchers to fetch, parse and update announcements from some source. These should be standalone individual scripts and linked to a `lien alias` command.

```clojure
; in update_announcements.clj
(defn -main "Fetch, parse and save new announcements")
```

Now that we have some data in database, we can proceed to create interface around it.

The `views.clj` should act as an interface between the user and application. The routes should connect to functions in `views.clj`.

```clojure
; in views.clj

(defn announcements "Render html page for seeing announcements")

(defn save-settings "Render and process user settings form")
```


We would need small standalone components functions which create tables or lists. We place them in `components.clj`. The components should be responsible for rendering data and not for "data processing". "Data Processing" should be completed before passing data to components.

```clojure
; in components.clj
(defn announcement "Render single announcement row")

(defn announcement-list "Render list of all announcements")
```


For rich filter, sorting and searching functionality we would need small functions which individually "process data". These functions should go in `data.clj`.

```clojure
; in data.clj

(defn sort "Modify query to handle sort in request")

(defn search "Modify query to handle search")

(defn user-preferences "Modify query as per user's saved settings")

(def update-user-default-sort "Set the users's default sort to given name")
```


To allow users to save their own preferences or data, we would need forms. These forms and defaults options should go in `rules.clj`. Rules is a standalone config file. It typically would not have functions but have all the application specific standalone variables.

```clojure
; in rules.clj

; Define fields and rules for settings form
(def SettingsForm "...")

; Define default filters
(def filters ["functions" "or" "values"])
```
