(ns utilities.forms.validations
  (:require [clojure.string :as s]))

; Sources: https://docs.djangoproject.com/en/2.1/_modules/django/core/validators/#validate_ipv46_address


(defn validate-required
  "Validates if the stripped value is a string of over 1 character"
  [value]
  (let [value (if (and (string? value)
                       (empty? (s/trim value)))
                nil
                value)]
    (if-not value
      {:error "This is a required field"}
      {:is-clean? true})))


(defn validate-email
  "Check is based on bouncer.validators which checks if there's
  only a single '@' and at least one point after the '@'
  "
  [value]
  (if (and (:is-clean? (validate-required value))
           (re-matches #"^[^@]+@[^@\\.]+[\\.].+" value))
    {:is-clean? true}
    {:error "Please provide a valid email"}))


(defn validate-min-length
  "Validates if the mininum length is satisfied"
  [length value]
  (if (>= (count value) length)
    {:is-clean? true}
    {:error (str "Value must be minimum " length " characters")}))


(defn validate-strong-password
  "Validates if the password is minimum 6 chars"
  [value]
  (if (:is-clean? (validate-min-length 6 value))
    {:is-clean? true}
    {:error "Password should be minimum 6 characters"}))


(defn validate-integer-str
  "Validates if the string is an integer"
  [value]
  (try
    (Integer. value)
    {:is-clean? true}
    (catch NumberFormatException e
      {:error "Please provide an integer value"})))


(defn validate-matching-password
  "Validates if the values match"
  [password confirm-password]
  (if-not (= password confirm-password)
    {:error "Passwords don't match"}
    {:is-clean? true}))
