(ns utilities.routing
  (:require [compojure.core :as c]
            [ring.util.codec :as codec]
            [stencil.core :as m]
            [stencil.parser :as p]))


; url paths is a map of name and compiled ast path
(defonce url-paths (atom {}))


(defn variable?
  "Returns true if given value is a mustache variable"
  [x]
  (record? x))


(defn get-url
  "Generates a function which provides url for given name and context"
  ([name] (get-url name {}))
  ([name context]
   (let [path (get @url-paths name)]
     (if-not path (throw (Exception.
                           (str "URL with name " name " not found\n"
                                "Available routes:" (keys @url-paths)))))
     (m/render path context))))


(defmacro url
  "Returns compojure url handler using stencil fillers"
  [url-name pattern f]
  (let [ast-path  (p/parse pattern)
        args      (map #(first (:name %)) (filter variable? ast-path))
        ; replace stencil variables with compujure-keyword syntax
        context   (apply merge (for [arg args]
                                 {arg arg}))
        path      (m/render ast-path context)
        arg-names (map #(symbol (name %)) args)
        request   (symbol "request")]
    (swap! url-paths assoc url-name ast-path)
    (if args
      `(c/ANY ~path [~@arg-names :as ~request] (~f ~request ~@arg-names))
      `(c/ANY ~path request (~f request)))))

(comment
  (macroexpand-1 '(url :foo "/foo/{{id}}/" #(fn [request id] "Hello"))))


(defn get-params
  "Returns get link after updating current query params with new data"
  [request data]
  (let [params   (:query-params request)
        params   (merge params data)]
    (str "?" (codec/form-encode params))))
