(ns utilities.db-test
  (:require [utilities.db :refer :all]
            [clojure.test :refer :all]))


(deftest test-builder
  (testing "various SQLs"
    (are [q r]
      (= (sql q) r)
      (query "books" {})  '("SELECT *\nFROM books")))

  (testing "no select"
    (let [q (query "company"
                   {:where ["id = ?" nil]})
          s (sql q)
          expected [(str "SELECT *\n"
                         "FROM company\n"
                         "WHERE id = ?") ""]]
      (is (= s expected))))

  (testing "query building"
    (let [q (query "users"
                   {:select "id, name"
                    :where ["email like ?" "foo%"]})
          s (sql q)
          expected [(str "SELECT id, name\n"
                         "FROM users\n"
                         "WHERE email like ?")
                    "foo%"]]
      (is (= s expected))))

  (testing "chained query building"
    (let [q (query "users"
                   {:select "id, name"
                    :where ["email like ?" "foo%"]})
          q (query q
                   {:select ["about AS ?" "user-about"]
                    :where ["about NOT LIKE ?" "%private%"]})
          s (sql q)
          expected [(str "SELECT id, name, about AS ?\n"
                         "FROM users\n"
                         "WHERE email like ? AND about NOT LIKE ?")
                    "user-about" "foo%" "%private%"]]
      (is (= s expected))))

  (testing "chained sort"
    (let [q (query "users" {:select "id"
                            :order-by "name asc, id desc"})
          q (query q {:select "name"})
          s (sql q)
          expected [(str "SELECT id, name\n"
                         "FROM users\n"
                         "ORDER BY name asc, id desc")]]
      (is (= s expected))))

  (testing "big sql"
    (let [q (query  "foobar"
                    {:select ["id, name, foo AS ?" "baz"]
                     :where [["some_col > ?" 1]
                             ["another_col > ?" 2]]
                     :group-by "bar"
                     :having ["? > ?" "baz" 5]})
          expected [(str "SELECT id, name, foo AS ?\n"
                         "FROM foobar\n"
                         "WHERE some_col > ? AND another_col > ?\n"
                         "GROUP BY bar\n"
                         "HAVING ? > ?")
                    "baz" 1 2 "baz" 5]]
      (is (= (sql q) expected))))

  (testing "nulls in where"
    (are [where expected]
      (= (sql (query "foo" {:where where}))
         (cons (str "SELECT *\n"
                    "FROM foo\n"
                    "WHERE " (first expected))
               (rest expected)))
      ["id <> ?" nil] ["id <> ?" ""]
      ["id <> ?" (:id nil)] ["id <> ?" ""])))
