(ns crux.moberg
  (:require [crux.codec :as c]
            [crux.memory :as mem]
            [crux.kv :as kv]
            [taoensso.nippy :as nippy])
  (:import [org.agrona DirectBuffer ExpandableDirectByteBuffer MutableDirectBuffer]
           org.agrona.io.DirectBufferInputStream
           crux.api.NonMonotonicTimeException
           java.util.function.Supplier
           [java.io Closeable DataInputStream DataOutput]
           java.nio.ByteOrder
           java.util.Date))

;; Based on
;; https://github.com/facebook/rocksdb/wiki/Implement-Queue-Service-Using-RocksDB

(set! *unchecked-math* :warn-on-boxed)

(def ^:private ^:const message-idx 1)
(def ^:private ^:const reverse-key-idx 2)

(def ^:private ^:const idx-id-size Byte/BYTES)

(def ^:private ^ThreadLocal topic-key-buffer-tl
  (ThreadLocal/withInitial
   (reify Supplier
     (get [_]
       (ExpandableDirectByteBuffer.)))))

(defn- topic-key ^org.agrona.DirectBuffer [topic message-id]
  (let [topic (c/->id-buffer topic)]
    (mem/limit-buffer
     (doto ^MutableDirectBuffer (.get topic-key-buffer-tl)
       (.putByte 0 message-idx)
       (.putBytes idx-id-size topic 0 c/id-size)
       (.putLong (+ idx-id-size c/id-size) message-id ByteOrder/BIG_ENDIAN))
     (+ idx-id-size c/id-size Long/BYTES))))

(def ^:private ^ThreadLocal reverse-key-key-buffer-tl
  (ThreadLocal/withInitial
   (reify Supplier
     (get [_]
       (ExpandableDirectByteBuffer.)))))

(defn- reverse-key-key ^org.agrona.DirectBuffer [topic k]
  (let [topic (c/->id-buffer topic)
        k (c/->id-buffer k)]
    (mem/limit-buffer
     (doto ^MutableDirectBuffer (.get reverse-key-key-buffer-tl)
       (.putByte 0 reverse-key-idx)
       (.putBytes idx-id-size topic 0 c/id-size)
       (.putBytes (+ idx-id-size c/id-size) k 0 (.capacity k)))
     (+ idx-id-size c/id-size c/id-size))))

(deftype CompactKVsAndKey [kvs key])

(defn- compact-topic-kvs+compact-k ^crux.moberg.CompactKVsAndKey [snapshot topic k new-message-k]
  (let [seek-k (reverse-key-key topic k)
        compact-k (kv/get-value snapshot seek-k)]
    (CompactKVsAndKey.
     (cond-> [[seek-k new-message-k]]
       compact-k (conj [compact-k c/empty-buffer]))
     compact-k)))

(def ^:private topic+date->count (atom {}))
(def ^:private ^:const seq-size 10)
(def ^:private ^:const max-seq-id (dec (bit-shift-left 1 seq-size)))

(defn- same-topic? [a b]
  (mem/buffers=? a b (+ idx-id-size c/id-size)))

(defn- message-id->message-time ^java.util.Date [^long message-id]
  (Date. (bit-shift-right message-id seq-size)))

(defn- message-key->message-id ^long [^DirectBuffer k]
  (.getLong k (+ idx-id-size c/id-size) ByteOrder/BIG_ENDIAN))

(defn end-message-id-offset ^long [kv topic]
  (with-open [snapshot (kv/new-snapshot kv)
              i (kv/new-iterator snapshot)]
    (let [seek-k (topic-key topic Long/MAX_VALUE)
          k (kv/seek i seek-k)]
      (if (and k (same-topic? seek-k k))
        (or (when-let [k ^DirectBuffer (kv/prev i)]
              (when (same-topic? k seek-k)
                (inc (message-key->message-id k))))
            1)
        (do (kv/store kv [[seek-k c/empty-buffer]])
            (end-message-id-offset kv topic))))))

(deftype SentMessage [^Date time ^long id topic])

(defn- now ^java.util.Date []
  (Date.))

(def ^:const detect-clock-drift? (not (Boolean/parseBoolean (System/getenv "CRUX_NO_CLOCK_DRIFT_CHECK"))))

(defn- next-message-id ^crux.moberg.SentMessage [kv topic]
  (let [message-time (now)
        seq (long (get (swap! topic+date->count
                              (fn [topic+date->count]
                                {message-time (inc (long (get topic+date->count message-time 0)))}))
                       message-time))
        message-id (bit-or (bit-shift-left (.getTime message-time) seq-size) seq)
        end-message-id (when detect-clock-drift?
                         (end-message-id-offset kv topic))]
    (cond
      (and detect-clock-drift? (< message-id (long end-message-id)))
      (throw (NonMonotonicTimeException.
              (str "Clock has moved backwards in time, message id: " message-id
                   " was generated using " (pr-str message-time)
                   " lowest valid next id: " end-message-id
                   " was generated using " (pr-str (message-id->message-time end-message-id)))))

      (> seq max-seq-id)
      (recur kv topic)

      :else
      (SentMessage. message-time message-id topic))))

(deftype Message [body topic ^long message-id ^Date message-time key headers])

(defn message->edn [^Message m]
  (cond-> {::body (.body m)
           ::topic (.topic m)
           ::message-id (.message-id m)
           ::message-time (.message-time m)}
    (.key m) (assoc ::key (.key m))
    (.headers m) (assoc ::headers (.headers m))))

(defn- nippy-thaw-message [topic message-id ^DirectBuffer buffer]
  (when (pos? (.capacity buffer))
    (with-open [in (DataInputStream. (DirectBufferInputStream. buffer))]
      (let [body (nippy/thaw-from-in! in)
            k (when (pos? (.available in))
                (nippy/thaw-from-in! in))
            headers (when (pos? (.available in))
                      (nippy/thaw-from-in! in))]
        (Message. body topic message-id (message-id->message-time message-id) k headers)))))

(def ^:private ^ThreadLocal send-buffer-tl
  (ThreadLocal/withInitial
   (reify Supplier
     (get [_]
       (ExpandableDirectByteBuffer.)))))

(defn sent-message->edn [^SentMessage m]
  {::message-id (.id m)
   ::message-time (.time m)
   ::topic (.topic m)})

(defn send-message
  (^SentMessage [kv topic v]
   (send-message kv topic nil v nil))
  (^SentMessage [kv topic k v]
   (send-message kv topic k v nil))
  (^SentMessage [kv topic k v headers]
   (with-open [^Closeable snapshot (if k
                                     (kv/new-snapshot kv)
                                     (reify Closeable
                                       (close [_])))]
     (let [id (next-message-id kv topic)
           message-time (.time id)
           message-id (.id id)
           message-k (topic-key topic message-id)
           compact-kvs+k (when k
                           (compact-topic-kvs+compact-k snapshot topic k message-k))]
       (kv/store kv (concat
                     (some->> compact-kvs+k (.kvs))
                     [[message-k
                       (mem/with-buffer-out
                         (.get send-buffer-tl)
                         (fn [^DataOutput out]
                           (nippy/freeze-to-out! out v)
                           (when (or k headers)
                             (nippy/freeze-to-out! out k)
                             (nippy/freeze-to-out! out headers)))
                         false)]]))
       (when-let [compact-k (some-> compact-kvs+k (.key))]
         (kv/delete kv [compact-k]))
       id))))

(defn next-message ^crux.moberg.Message [i topic]
  (let [seek-k (topic-key topic 0)]
    (when-let [k ^DirectBuffer (kv/next i)]
      (when (same-topic? k seek-k)
        (or (nippy-thaw-message topic (message-key->message-id k) (kv/value i))
            (recur i topic))))))

(defn seek-message
  (^crux.moberg.Message [i topic]
   (seek-message i topic nil))
  (^crux.moberg.Message [i topic message-id]
   (let [seek-k (topic-key topic (or message-id 0))]
     (when-let [k ^DirectBuffer (kv/seek i seek-k)]
       (when (same-topic? k seek-k)
         (or (nippy-thaw-message topic (message-key->message-id k) (kv/value i))
             (next-message i topic)))))))
