A Leiningen plugin to manage database migrations in Clojure projects.

Inspired by database migrations in Ruby on Rails.

## Features

* Minimal set of templates to generate migrations
* Easily add your own template migrations
* No special DSL - just plain Clojure code that runs within the context of your project
* Apply and rollback migrations from the command line


## Usage

### Setup

Add **Kapooya** as a plugin to your project.clj file:

    :plugins [[kapooya/kap "0.1.4"]]


Add a :kapooya entry to your :profiles map. This can be done in your project.clj file:

    :profiles { :dev { :kapooya { :environment "development"
                                   :db-spec { :subprotocol "postgresql"
                                              :subname "//localhost/mydatabase"
                                              :user "root" }}}}

Alternatively, you can use a profiles.clj file (see [profiles.clj.sample](https://github.com/bayan/kapooya/blob/master/profiles.clj.sample) for an example).

The `:kapooya` key points to a map with two entries:

`:environment` Optional, defaults to 'development'.
Multiple databases can be managed with **Kapooya**. A common use case is to have a separate testing, development and production databases.
This key identifies the name of the environment associated with the current profile.

`:db-spec` Mandatory, must be specified.
This value is passed directly to the migration functions and is meant to be used to connect to the database.
Currently, PostgreSQL is supported using java.jdbc.

### Creating a database

**Kapooya** does not create a database. The database should already exist and be accessible using the `:db-spec` map.

I.e. Use the following command to create a PostgreSQL database:

```
createdb mydatabase
```


### Commands

Once the plugin has been setup in the project.clj file you are ready to run the following commands:

`lein kap help` Print a short help message (default)

`lein kap init` Creates directories that are required by **Kapooya** (Nondestructive).

#### Generating new migrations

`lein kap generate custom` Creates a new migration to be completed with some custom code.

`lein kap generate create-table table-name` Creates a new migration to create a table.

`lein kap generate drop-table table-name` Creates a new migration to drop a table.

`lein kap generate add-column table-name column-name [column-type]` Creates a new migration to add a column to a table. Column type defaults to `character varying(255)`.

`lein kap generate drop-column table-name column-name [column-type]` Creates a new migration to drop a column from a table.

`lein kap generate create-index table-name column-name` Creates a new migration to create an index.

`lein kap generate drop-index table-name column-name` Creates a new migration to drop an index.

The `custom`, `create-table` and `drop-table` generators make use of an optional :example parameter. When the arguments include :example, then some sample code is included in the generated migration file. For example, running the following command:

`lein kap generate custom do many things :example`

Should generate a migration with contents similar to the following:

```clojure
(ns db.migrations.20131104112743-do-many-things
  (:require [clojure.java.jdbc :as jdbc]
            [clojure.java.jdbc.ddl :as ddl]))

(defn up [db-spec]
  (jdbc/execute! db-spec ["INSERT SQL VALID CODE HERE"]))

(defn down [db-spec]
  (jdbc/execute! db-spec ["INSERT SQL VALID CODE HERE"]))

```

Omitting the `:example` argument as in the following command:

`lein kap generate custom do many things`

Should instead generate a migration with contents similar to the following:

```clojure
(ns db.migrations.20131104112743-do-many-things
  (:require [clojure.java.jdbc :as jdbc]
            [clojure.java.jdbc.ddl :as ddl]))

(defn up [db-spec]
  )

(defn down [db-spec]
  )

```

#### Running migrations

`lein kap migrate` Run the `up` method for all outstanding migrations sorted by filename (which starts with a timestamp).

`lein kap rollback` Run the `down` method for the most recent migration.


## DIY Templates

You can create custom templates that can be used to generate new migrations. Simply create a .clj file and place it in the projects 'db/templates' directory.

The file name for your template should be a valid clojure filename and is used to locate the template when generating migrations. The first argument after the generate command should match the file name (sans the `.clj` extension).

**Kapooya** templates are Clojure files written with the [Mustache](http://mustache.github.com) templating language (via [Stencil](https://github.com/davidsantiago/stencil)).

Refer to the [supplied templates](https://github.com/bayan/kapooya/tree/master/resources/templates) as examples to refer to when creating your own.

The following variables are made available when processing the template with [Mustache](http://mustache.github.com) (via [Stencil](https://github.com/davidsantiago/stencil)):
* `namespace` Used to set the namespace of the resulting migration file. **Kapooya** migrations require the namespace declaration to be specified at the top of the file (i.e. `(ns {{namespace}} ...)`).
* `example` A boolean value that is true if the command to use the generator passed in the optional `:example` argument, and false otherwise.
* `arg0`, `arg1`, `arg2`, etc. These are the additional arguments that are supplied when using the migration generator command.


## Dependencies

**Kapooya** uses [clojure.java.jdbc](https://github.com/clojure/java.jdbc) in the existing migration templates. The version used in the project.clj dependencies will be used if it exists. Otherwise, the dependency will be added at runtime when running any migration command (i.e. `migrate` or `rollback`).

Similarly, the [PostgreSQL JDBC Driver](http://jdbc.postgresql.org/) is added as a dependency at runtime if it is not already present in the projects list of dependencies.


## Supported databases

PostgreSQL

## TODO

* Tutorial
* Tests
* Support and templates for other databases
  * Relational
    * MySQL
    * SQLite
  * Non-relational
    * Cassandra
* Allow more than one database to be used on a project


## License

Copyright © 2013

Distributed under the Eclipse Public License, the same as Clojure.
