;; sir_philip_sidney.clj
;;
;; by Kyle Harrington, kyleh@cs.brandeis.edu, 2011.

(ns examples.sir-philip-sidney
  (:use [clojush :exclude '-main]))

;;
;; The Sir Philip Sidney game involves a donor and a beneficiary. The
;;   beneficiary is either thirsty or not, and can signal to indicate
;;   whether zhe is thirsty. The donor then decides whether to give a
;;   resource to improve the probability of the beneficiary's survival.
;;   This donation reduces the probability of survival of the donor.
;;   Scoring involves the relatedness of the donor and beneficiary.
;;

(define-registered am_thirsty?
  (fn [state]
    (if (:beneficiary state)
      (push-item (:thirsty state) :boolean state)
      state)))

(define-registered signal_thirst
  (fn [state]
    (if (:beneficiary state)
      (assoc state
        :signal-thirst true)
      state)))

(define-registered recieved_signal?
  (fn [state]
    (if (:donor state)
      (push-item (:recieved-signal? state) :boolean state)
      state)))

(define-registered donate_resource
  (fn [state]
    (if (:donor state)
      (assoc state
        :donate-resource true)
      state)))

(define-registered if_thirsty
  (fn [state]
    (if (not (or (:donor state)
                 (empty? (rest (:exec state)))))
      (push-item (if (:thirsty state)
                   (first (:exec state))
                   (first (rest (:exec state))))
        :exec
        (pop-item :boolean (pop-item :exec (pop-item :exec state))))
      state)))

(define-registered if_recieved_signal
  (fn [state]
    (if (not (or (:beneficiary state)
                 (empty? (rest (:exec state)))))
      (push-item (if (:recieved-signal? state)
                   (first (:exec state))
                   (first (rest (:exec state))))
        :exec
        (pop-item :boolean (pop-item :exec (pop-item :exec state))))
      state)))

(def minimum-atom-generators '(signal_thirst donate_resource if_thirsty if_recieved_signal))

(def atom-generators (concat minimum-atom-generators
                             '(exec_if boolean_not boolean_or boolean_and boolean_eq am_thirsty? recieved_signal?
                               boolean_rot boolean_swap boolean_dup boolean_pop true false)))

;; Behaviors and risk values
;;   beneficiary never signals if: thirsty-risk > signal-risk + relatedness * donation-risk
;;   beneficiary always signals if: healthy-risk < signal-risk + relatedness * donation-risk
;;   donor never donates if: thirsty-risk > donation-risk / relatedness
;;   donor always gives if: healthy-risk < donation-risk / relatedness


(defn game-creator
  "Create an game playing function. Plays all 4 combinations of the game with the specified
risks. Note: the scores here are ERROR, not survival as is the case in the formal description
of the game.
In the classical formulation the variable names are:
a, thirsty-risk
b, healthy-risk
c, signal-risk
d, donation-risk"
  [thirsty-risk healthy-risk signal-risk donation-risk]
  (fn play-game    
    [player1 player2]
    (doall (for [donor (list player1 player2)
                 thirsty '(true false)]
             (let [;; Beneficiary's play
                   beneficiary (if (= donor player1) player2 player1)
                   beneficiary-state (run-push beneficiary (assoc (make-push-state)
                                                             :beneficiary true
                                                             :thirsty thirsty))
                   beneficiary-signal (:signal-thirst beneficiary-state)
                   ;; Donor's play
                   donor-state (run-push donor (assoc (make-push-state)
                                                 :donor true
                                                 :recieved-signal? beneficiary-signal))
                   donation (:donate-resource donor-state)
                   ;; Relatedness
                   relatedness (overlap player1 player2)
                   ;; Role scoring
                   beneficiary-score (+ (cond (and thirsty donation) 0
                                              (and thirsty (not donation)) thirsty-risk
                                              (and (not thirsty) donation) 0
                                              (and (not thirsty) (not donation)) healthy-risk)
                                        (if beneficiary-signal signal-risk 0))
                   donor-score (if donation donation-risk 0)
                   ;; Player scoring
                   player1-score (if (= donor player1) donor-score beneficiary-score)
                   player2-score (if (= donor player1) beneficiary-score donor-score)]
               [(+ player1-score (* relatedness player2-score))
                (+ player2-score (* relatedness player1-score))])))))

(def game-tournament-size 10)

(defn evaluate-individual-with-game
  "Returns the given individual with errors and total-errors, computing them if necessary."
  [i error-function rand-gen population tournament-size]
; (println "XXXXXX") (flush) ;***
  (binding [thread-local-random-generator rand-gen]
    (let [tournament (doall (repeatedly game-tournament-size #(lrand-nth population)))
          p (:program i)
          game-results (doall (mapcat #(error-function p (:program %)) tournament))
          e (doall (map first game-results))
          te (keep-number-reasonable (reduce + e))]
;(println te)(flush) ;***
      (make-individual :program p :errors e :total-error te 
        :history (if maintain-histories (cons te (:history i)) (:history i))
        :ancestors (:ancestors i)))))

(defn gamegp
  "The top-level routine of pushgp."
  [& {:keys [error-function error-threshold population-size max-points atom-generators max-generations
             max-mutations mutation-probability mutation-max-points crossover-probability 
             simplification-probability tournament-size report-simplifications final-report-simplifications
             reproduction-simplifications trivial-geography-radius decimation-ratio decimation-tournament-size
             evalpush-limit evalpush-time-limit node-selection-method node-selection-leaf-probability
             node-selection-tournament-size pop-when-tagging gaussian-mutation-probability 
             gaussian-mutation-per-number-mutation-probability gaussian-mutation-standard-deviation
	     reuse-errors problem-specific-report use-indirect-tagging tag-limit decimation-method
             autoconstructive-mutation-probability autoconstructive-crossover-probability
             use-single-thread random-seed]
      :or {error-function (fn [p] '(0)) ;; pgm -> list of errors (1 per case)
           error-threshold 0
           population-size 1000
           max-points 50
           atom-generators (concat @registered-instructions
                             (list 
                               (fn [] (lrand-int 100))
                               (fn [] (lrand))))
           max-generations 1001
           mutation-probability 0.45
           mutation-max-points 20
           crossover-probability 0.45
           simplification-probability 0.0
           tournament-size 7
           report-simplifications 100
           final-report-simplifications 1000
           reproduction-simplifications 1
           trivial-geography-radius 0
           decimation-ratio 1
           decimation-tournament-size 2;; When using pareto-decimate or clone-decimate this serves as the max # of attempts
           evalpush-limit 150
           evalpush-time-limit 0
           node-selection-method :unbiased
           node-selection-leaf-probability 0.1
           node-selection-tournament-size 2
           pop-when-tagging true
           gaussian-mutation-probability 0.0
           gaussian-mutation-per-number-mutation-probability 0.5
           gaussian-mutation-standard-deviation 0.1
	   reuse-errors true
	   problem-specific-report default-problem-specific-report
           use-indirect-tagging false
           tag-limit 1000
           decimation-method :total-error ;; :pareto, :clone, or :crowding
           autoconstructive-mutation-probability 0
           autoconstructive-crossover-probability 0
           use-single-thread false
           random-seed 0
           }}]
  (let [atom-generators (concat atom-generators
                                (when (or (> autoconstructive-mutation-probability 0)
                                          (> autoconstructive-crossover-probability 0))
                                  (list 'zip_down 'zip_left 'zip_right 'zip_rand 'zip_root 'zip_rloc 'zip_rrloc))
                                (when (> autoconstructive-crossover-probability 0)
                                  (list 'zip_swap 'zip_subtree_swap)))]
    ;; set globals from parameters
    (reset! global-atom-generators atom-generators)
    (reset! global-max-points-in-program max-points)
    (reset! global-evalpush-limit evalpush-limit)
    (reset! global-evalpush-time-limit evalpush-time-limit)
    (reset! global-node-selection-method node-selection-method)
    (reset! global-node-selection-leaf-probability node-selection-leaf-probability)
    (reset! global-node-selection-tournament-size node-selection-tournament-size)
    (reset! global-pop-when-tagging pop-when-tagging)
    (reset! global-reuse-errors reuse-errors)
    (reset! global-use-indirect-tagging use-indirect-tagging)
    (reset! global-tag-limit tag-limit)
    (printf "\nStarting PushGP run.\n\n") (flush)
    (print-params 
     (error-function error-threshold population-size max-points atom-generators max-generations 
                     mutation-probability mutation-max-points crossover-probability
                     simplification-probability gaussian-mutation-probability 
                     gaussian-mutation-per-number-mutation-probability gaussian-mutation-standard-deviation
                     tournament-size report-simplifications final-report-simplifications
                     trivial-geography-radius decimation-ratio decimation-tournament-size evalpush-limit
                     evalpush-time-limit node-selection-method node-selection-tournament-size
                     node-selection-leaf-probability pop-when-tagging reuse-errors use-indirect-tagging
                     tag-limit decimation-method autoconstructive-mutation-probability
                     autoconstructive-crossover-probability use-single-thread random-seed
                     ))
    (printf "\nGenerating initial population...\n") (flush)
    (let [pop-agents (vec (doall (for [_ (range population-size)] 
                                   ((if use-single-thread atom agent)
                                    (make-individual 
                                     :program (random-code max-points atom-generators))
                                    :error-handler (fn [agnt except] (println except))))))
          child-agents (vec (doall (for [_ (range population-size)]
                                     ((if use-single-thread atom agent)
                                      (make-individual)
                                      :error-handler (fn [agnt except] (println except))))))
          rand-gens (vec (doall (for [k (range population-size)]
                                  (java.util.Random. (+ random-seed k)))))]
      (loop [generation 0]
        (printf "\n\n-----\nProcessing generation: %s\nComputing errors..." generation) (flush)
        (let [population (vec (doall (map deref pop-agents)))]
          (dorun (map #((if use-single-thread swap! send) % evaluate-individual-with-game error-function %2 population tournament-size)
                      pop-agents rand-gens))
          (when-not use-single-thread (apply await pop-agents))) ;; SYNCHRONIZE ; might this need a dorun?
        (printf "\nDone computing errors.") (flush)
        
        ;; some debugging code trying to track down nil agent results.... leaving in case not fixed
                                        ;(println (map :total-error (vec (doall (map deref pop-agents)))))(flush) ;***
                                        ;(loop [ers (map :total-error (vec (doall (map deref pop-agents))))] ;***
                                        ;  (when (some not ers) 
                                        ;    (println (map :total-error (vec (doall (map deref pop-agents)))))(flush) 
                                        ;    (recur (map :total-error (vec (doall (map deref pop-agents)))))))
        
        ;; report and check for success
        (let [best (report (vec (doall (map deref pop-agents))) generation error-function 
                           report-simplifications problem-specific-report)]
          (if (<= (:total-error best) error-threshold)
            (do (printf "\n\nSUCCESS at generation %s\nSuccessful program: %s\nErrors: %s\nTotal error: %s\nHistory: %s\nSize: %s\n\n"
                        generation (not-lazy (:program best)) (not-lazy (:errors best)) (:total-error best) 
                        (not-lazy (:history best)) (count-points (:program best)))
                (when print-ancestors-of-solution
                  (printf "\nAncestors of solution:\n")
                  (println (:ancestors best)))
                (auto-simplify best error-function final-report-simplifications true 500))
            (do (if (>= generation max-generations)
                  (printf "\nFAILURE\n")
                  (do (printf "\nProducing offspring...") (flush)
                      (let [pop ((cond (= decimation-method :pareto) pareto-decimate
                                       (= decimation-method :crowding) crowding-decimate
                                       (= decimation-method :clone) clone-decimate
                                       :else decimate)
                                  (vec (doall (map deref pop-agents))) 
                                                 (int (* decimation-ratio population-size))
                                                 decimation-tournament-size 
                                                 trivial-geography-radius)]
                        (printf "\nNumber of individuals removed by decimation: %s\n" (- (count pop-agents) (count pop)))
                        (dotimes [i population-size]
                          ((if use-single-thread swap! send)
                           (nth child-agents i) 
                                breed i (nth rand-gens i) pop error-function population-size max-points atom-generators 
                                mutation-probability mutation-max-points crossover-probability 
                                simplification-probability tournament-size reproduction-simplifications 
                                trivial-geography-radius gaussian-mutation-probability 
                                gaussian-mutation-per-number-mutation-probability gaussian-mutation-standard-deviation
                                autoconstructive-mutation-probability autoconstructive-crossover-probability)))
                      (when-not use-single-thread (apply await child-agents)) ;; SYNCHRONIZE
                      (printf "\nInstalling next generation...") (flush)
                      (dotimes [i population-size]
                        ((if use-single-thread swap! send)
                         (nth pop-agents i) (fn [av] (deref (nth child-agents i)))))
                      (when-not use-single-thread (apply await pop-agents)) ;; SYNCHRONIZE
                      (recur (inc generation)))))))))))


;(gamegp :atom-generators minimum-atom-generators :error-function error-fn :use-single-thread true :report-simplifications 0 :final-report-simplifications 0 :max-points 200 :population-size 2000 :max-generations 10001)