(ns experiments.ACE-DSOAR
  (:require [clojure.contrib.duck-streams :as ds]
            [clojure.string :as s])
  (:use; [popen]
        [clojure.contrib.math]
        [clojure.pprint]))

(defn variation-parameter-map
  [& {:keys [crossover-probability mutation-probability gaussian-mutation-probability simplification-probability
             autoconstructive-mutation-probability autoconstructive-crossover-probability autoconstructive-tournament-size]
      :or {crossover-probability 0
           mutation-probability 0
           gaussian-mutation-probability 0
           simplification-probability 0
           autoconstructive-mutation-probability 0
           autoconstructive-crossover-probability 0
           autoconstructive-tournament-size 2}}]
  {:crossover-probability crossover-probability
   :mutation-probability mutation-probability
   :guassian-mutation-probability gaussian-mutation-probability
   :simplification-probability simplification-probability
   :autoconstructive-mutation-probability autoconstructive-mutation-probability
   :autoconstructive-crossover-probability autoconstructive-crossover-probability
   :autoconstructive-tournament-size autoconstructive-tournament-size})

(defn argmap-to-argstr
  "Convert a map of arguments into a command line string for lein run."
  [m]
  (s/join " " (doall (map #(str "--" (s/replace (first %) ":" "") " " (second %)) (seq m)))))

(defn argmap-to-logname
  "Convert a map into an appropriate log filename."
  [m uuid]
  (str uuid (s/join "_" (doall (map #(str (second %)) (seq m))))))

(def hpc-directory "ACE_DSOAR_6")

(defn hpc-script
  "Take a sequence of commands and create a job for each command"
  [cmds script-name]
  (ds/with-out-writer script-name
    (doseq [cmd cmds]
      (println "echo \"cd" hpc-directory ";" cmd "\" | qsub -ckpt reloc -N DEMO_ACE_DSOAR"))))

(defn toc-script
  "Write the table of contents to file."
  [toc filename]
  (ds/with-out-writer filename
    (doseq [entry toc]
      (println entry))))

(defn analysis-script
  "Write an analysis script."
  [ah filename]
  (let [names (for [[el name] ah] (s/join "*" name))
        ah    (for [[el name] ah] (s/join "*" el))]
    (ds/with-out-writer filename
      (println "for f in *.gz; do gunzip $f; done;")
      (println (str "lein run -m analysis "
                    (s/join " " (interleave (map #(str "--" %) names)
                                            (map #(str "-/Users/kyle/clj-workspace/Clojush/results/" hpc-directory "/" %) ah))))))))

(def runID (atom 0))
(def table-of-contents (atom []))
(def analysis-handler (atom []))

(def commands
  (doall (for [num-repeats (range 100)
               other-parameters [{:use-single-thread true}]
               tag-parameters [{:use-tags true :use-indirect-tagging false :tag-limit 1000}]
               variation-parameters [(variation-parameter-map :crossover-probability 0.8 :mutation-probability 0.1)
                                     (variation-parameter-map :autoconstructive-crossover-probability 0.8 :autoconstructive-mutation-probability 0.2
                                                              :autoconstructive-tournament-size 2)
                                     (variation-parameter-map :autoconstructive-crossover-probability 0.5 :autoconstructive-mutation-probability 0.5
                                                              :autoconstructive-tournament-size 2)
                                     (variation-parameter-map :autoconstructive-crossover-probability 0.8 :autoconstructive-mutation-probability 0.2
                                                              :autoconstructive-tournament-size 5)
                                     (variation-parameter-map :autoconstructive-crossover-probability 0.5 :autoconstructive-mutation-probability 0.5
                                                              :autoconstructive-tournament-size 5)
                                     (variation-parameter-map :autoconstructive-crossover-probability 0.8 :autoconstructive-mutation-probability 0.2
                                                              :autoconstructive-tournament-size 9)]                                     
               decimation-parameters [{}]
;                                      {:decimation-method :clone :decimation-ratio 0.9 :decimation-tournament-size 5}]
               problem-parameters [{:num-maps 3 :num-cols 6 :num-rows 6 :num-obstacles 5}
                                   {:num-maps 3 :num-cols 8 :num-rows 8 :num-obstacles 8}
                                   {:num-maps 3 :num-cols 10 :num-rows 10 :num-obstacles 14}]]
           (let [argmap (merge other-parameters tag-parameters variation-parameters decimation-parameters problem-parameters {:random-seed (System/nanoTime)})
                 outputname (argmap-to-logname argmap (str "DSOARACE" @runID "_"))]
             (when (zero? num-repeats)
               (reset! analysis-handler (cons (let [ms (seq (merge other-parameters tag-parameters variation-parameters
                                                                   decimation-parameters problem-parameters
                                                                   {:random-seed (System/nanoTime)}))]
                                                [["DSOARACE"
                                                 (s/join "_" (doall (map #(str (second %))
                                                                         (take-while #(not= (first %) :random-seed) ms))))
                                                 (s/join "_" (doall (map #(str (second %))
                                                                         (rest (drop-while #(not= (first %) :random-seed) ms)))))]
                                                ["DSOARACE"
                                                 (s/join "_" (doall (map #(str (first %) "=" (second %))
                                                                         (take-while #(not= (first %) :random-seed) ms))))
                                                 (s/join "_" (doall (map #(str (first %) "=" (second %))
                                                                         (rest (drop-while #(not= (first %) :random-seed) ms)))))]])
                                              @analysis-handler)))
             (reset! table-of-contents (cons (assoc argmap :runID @runID) @table-of-contents))
             (swap! runID inc)             
             (str "lein run -m examples.dsoar " (argmap-to-argstr argmap)
                  " > " outputname "; gzip " outputname)))))

(hpc-script commands (str hpc-directory ".sh"))
(toc-script @table-of-contents (str hpc-directory ".toc"))
(analysis-script @analysis-handler (str hpc-directory "_analysis.sh"))

; rsync -az clj-workspace/Clojush/* kyleh@hpc64.brandeis.edu:ACE-DSOAR/
; sh ACE_DSOAR_1.sh

;  lein run -m examples.dsoar --use-tags true --crossover-probability 0 --autoconstructive-crossover-probability 0.45 --use-indirect-tagging true --mutation-probability 0.01 --autoconstructive-mutation-probability 0.45 --decimation-method :clone --decimation-ratio 0.9 --decimation-tournament-size 5 --num-maps 4 --num-cols 6 --num-rows 6 --num-obstacles 8