/*
 * Decompiled with CFR 0.152.
 */
package org.jblas;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import org.jblas.ComplexFloat;
import org.jblas.FloatMatrix;
import org.jblas.JavaBlas;
import org.jblas.MatrixFunctions;
import org.jblas.NativeBlas;
import org.jblas.SimpleBlas;
import org.jblas.exceptions.SizeException;

public class ComplexFloatMatrix {
    public int rows;
    public int columns;
    public int length;
    public float[] data = null;

    public ComplexFloatMatrix(int newRows, int newColumns, float ... newData) {
        this.rows = newRows;
        this.columns = newColumns;
        this.length = this.rows * this.columns;
        if (newData.length != 2 * newRows * newColumns) {
            throw new IllegalArgumentException("Passed data must match matrix dimensions.");
        }
        this.data = newData;
    }

    public ComplexFloatMatrix(int newRows, int newColumns) {
        this(newRows, newColumns, new float[2 * newRows * newColumns]);
    }

    public ComplexFloatMatrix() {
        this(0, 0, null);
    }

    public ComplexFloatMatrix(int len) {
        this(len, 1, new float[2 * len]);
    }

    public ComplexFloatMatrix(float[] newData) {
        this(newData.length / 2);
        this.data = newData;
    }

    public ComplexFloatMatrix(ComplexFloat[] newData) {
        this(newData.length);
        for (int i = 0; i < newData.length; ++i) {
            this.put(i, newData[i]);
        }
    }

    public ComplexFloatMatrix(FloatMatrix m) {
        this(m.rows, m.columns);
        NativeBlas.scopy(m.length, m.data, 0, 1, this.data, 0, 2);
    }

    public ComplexFloatMatrix(FloatMatrix real, FloatMatrix imag) {
        this(real.rows, real.columns);
        real.assertSameSize(imag);
        if (real != null) {
            NativeBlas.scopy(this.length, real.data, 0, 1, this.data, 0, 2);
        }
        if (imag != null) {
            NativeBlas.scopy(this.length, imag.data, 0, 1, this.data, 1, 2);
        }
    }

    public ComplexFloatMatrix(String filename) throws IOException {
        this.load(filename);
    }

    public ComplexFloatMatrix(float[][] data) {
        this(data.length, data[0].length);
        int r;
        for (r = 0; r < this.rows; ++r) {
            assert (data[r].length == this.columns);
        }
        for (r = 0; r < this.rows; ++r) {
            for (int c = 0; c < this.columns; ++c) {
                this.put(r, c, data[r][c]);
            }
        }
    }

    public static ComplexFloatMatrix zeros(int rows, int columns) {
        return new ComplexFloatMatrix(rows, columns);
    }

    public static ComplexFloatMatrix zeros(int length) {
        return ComplexFloatMatrix.zeros(length, 1);
    }

    public static ComplexFloatMatrix ones(int rows, int columns) {
        ComplexFloatMatrix m = new ComplexFloatMatrix(rows, columns);
        for (int i = 0; i < rows * columns; ++i) {
            m.put(i, 1.0f);
        }
        return m;
    }

    public static ComplexFloatMatrix ones(int length) {
        return ComplexFloatMatrix.ones(length, 1);
    }

    public static ComplexFloatMatrix diag(ComplexFloatMatrix x) {
        ComplexFloatMatrix m = new ComplexFloatMatrix(x.length, x.length);
        for (int i = 0; i < x.length; ++i) {
            m.put(i, i, x.get(i));
        }
        return m;
    }

    public static ComplexFloatMatrix scalar(float s) {
        ComplexFloatMatrix m = new ComplexFloatMatrix(1, 1);
        m.put(0, 0, s);
        return m;
    }

    public boolean isScalar() {
        return this.length == 1;
    }

    public ComplexFloat scalar() {
        return this.get(0);
    }

    public static ComplexFloatMatrix concatHorizontally(ComplexFloatMatrix A, ComplexFloatMatrix B) {
        if (A.rows != B.rows) {
            throw new SizeException("Matrices don't have same number of rows.");
        }
        ComplexFloatMatrix result = new ComplexFloatMatrix(A.rows, A.columns + B.columns);
        SimpleBlas.copy(A, result);
        NativeBlas.ccopy(B.length, B.data, 0, 1, result.data, A.length, 1);
        return result;
    }

    public static ComplexFloatMatrix concatVertically(ComplexFloatMatrix A, ComplexFloatMatrix B) {
        if (A.columns != B.columns) {
            throw new SizeException("Matrices don't have same number of columns.");
        }
        ComplexFloatMatrix result = new ComplexFloatMatrix(A.rows + B.rows, A.columns);
        for (int i = 0; i < A.columns; ++i) {
            NativeBlas.ccopy(A.rows, A.data, A.index(0, i), 1, result.data, result.index(0, i), 1);
            NativeBlas.ccopy(B.rows, B.data, B.index(0, i), 1, result.data, result.index(A.rows, i), 1);
        }
        return result;
    }

    public ComplexFloatMatrix get(int[] indices) {
        ComplexFloatMatrix result = new ComplexFloatMatrix(indices.length);
        for (int i = 0; i < indices.length; ++i) {
            result.put(i, this.get(indices[i]));
        }
        return result;
    }

    public ComplexFloatMatrix get(int r, int[] indices) {
        ComplexFloatMatrix result = new ComplexFloatMatrix(1, indices.length);
        for (int i = 0; i < indices.length; ++i) {
            result.put(i, this.get(r, indices[i]));
        }
        return result;
    }

    public ComplexFloatMatrix get(int[] indices, int c) {
        ComplexFloatMatrix result = new ComplexFloatMatrix(indices.length, c);
        for (int i = 0; i < indices.length; ++i) {
            result.put(i, this.get(indices[i], c));
        }
        return result;
    }

    public ComplexFloatMatrix get(int[] rindices, int[] cindices) {
        ComplexFloatMatrix result = new ComplexFloatMatrix(rindices.length, cindices.length);
        for (int i = 0; i < rindices.length; ++i) {
            for (int j = 0; j < cindices.length; ++j) {
                result.put(i, j, this.get(rindices[i], cindices[j]));
            }
        }
        return result;
    }

    public ComplexFloatMatrix get(ComplexFloatMatrix indices) {
        return this.get(indices.findIndices());
    }

    public ComplexFloatMatrix get(int r, ComplexFloatMatrix indices) {
        return this.get(r, indices.findIndices());
    }

    public ComplexFloatMatrix get(ComplexFloatMatrix indices, int c) {
        return this.get(indices.findIndices(), c);
    }

    public ComplexFloatMatrix get(ComplexFloatMatrix rindices, ComplexFloatMatrix cindices) {
        return this.get(rindices.findIndices(), cindices.findIndices());
    }

    private void checkLength(int l) {
        if (this.length != l) {
            throw new SizeException("Matrix does not have the necessary length (" + this.length + " != " + l + ").");
        }
    }

    private void checkRows(int r) {
        if (this.rows != r) {
            throw new SizeException("Matrix does not have the necessary length (" + this.length + " != " + r + ").");
        }
    }

    private void checkColumns(int c) {
        if (this.columns != c) {
            throw new SizeException("Matrix does not have the necessary length (" + this.length + " != " + c + ").");
        }
    }

    public ComplexFloatMatrix put(int[] indices, ComplexFloatMatrix x) {
        if (x.isScalar()) {
            return this.put(indices, x.scalar());
        }
        x.checkLength(indices.length);
        for (int i = 0; i < indices.length; ++i) {
            this.put(indices[i], x.get(i));
        }
        return this;
    }

    public ComplexFloatMatrix put(int r, int[] indices, ComplexFloatMatrix x) {
        if (x.isScalar()) {
            return this.put(r, indices, x.scalar());
        }
        x.checkColumns(indices.length);
        for (int i = 0; i < indices.length; ++i) {
            this.put(r, indices[i], x.get(i));
        }
        return this;
    }

    public ComplexFloatMatrix put(int[] indices, int c, ComplexFloatMatrix x) {
        if (x.isScalar()) {
            return this.put(indices, c, x.scalar());
        }
        x.checkRows(indices.length);
        for (int i = 0; i < indices.length; ++i) {
            this.put(indices[i], c, x.get(i));
        }
        return this;
    }

    public ComplexFloatMatrix put(int[] rindices, int[] cindices, ComplexFloatMatrix x) {
        if (x.isScalar()) {
            return this.put(rindices, cindices, x.scalar());
        }
        x.checkRows(rindices.length);
        x.checkColumns(cindices.length);
        for (int i = 0; i < rindices.length; ++i) {
            for (int j = 0; j < cindices.length; ++j) {
                this.put(rindices[i], cindices[j], x.get(i, j));
            }
        }
        return this;
    }

    public ComplexFloatMatrix put(int[] indices, float v) {
        for (int i = 0; i < indices.length; ++i) {
            this.put(indices[i], v);
        }
        return this;
    }

    public ComplexFloatMatrix putReal(int[] indices, float v) {
        return this.put(indices, v);
    }

    public ComplexFloatMatrix putImag(int[] indices, float v) {
        for (int i = 0; i < indices.length; ++i) {
            this.putImag(indices[i], v);
        }
        return this;
    }

    public ComplexFloatMatrix put(int[] indices, ComplexFloat v) {
        for (int i = 0; i < indices.length; ++i) {
            this.put(indices[i], v);
        }
        return this;
    }

    public ComplexFloatMatrix put(int r, int[] indices, float v) {
        for (int i = 0; i < indices.length; ++i) {
            this.put(r, indices[i], v);
        }
        return this;
    }

    public ComplexFloatMatrix putReal(int r, int[] indices, float v) {
        return this.put(r, indices, v);
    }

    public ComplexFloatMatrix putImag(int r, int[] indices, float v) {
        for (int i = 0; i < indices.length; ++i) {
            this.putImag(r, indices[i], v);
        }
        return this;
    }

    public ComplexFloatMatrix put(int r, int[] indices, ComplexFloat v) {
        for (int i = 0; i < indices.length; ++i) {
            this.put(r, indices[i], v);
        }
        return this;
    }

    public ComplexFloatMatrix put(int[] indices, int c, float v) {
        for (int i = 0; i < indices.length; ++i) {
            this.put(indices[i], c, v);
        }
        return this;
    }

    public ComplexFloatMatrix putReal(int[] indices, int c, float v) {
        return this.put(indices, c, v);
    }

    public ComplexFloatMatrix putImag(int[] indices, int c, float v) {
        for (int i = 0; i < indices.length; ++i) {
            this.putImag(indices[i], c, v);
        }
        return this;
    }

    public ComplexFloatMatrix put(int[] indices, int c, ComplexFloat v) {
        for (int i = 0; i < indices.length; ++i) {
            this.put(indices[i], c, v);
        }
        return this;
    }

    public ComplexFloatMatrix put(int[] rindices, int[] cindices, float v) {
        for (int i = 0; i < rindices.length; ++i) {
            for (int j = 0; j < cindices.length; ++j) {
                this.put(rindices[i], cindices[j], v);
            }
        }
        return this;
    }

    public ComplexFloatMatrix putReal(int[] rindices, int[] cindices, float v) {
        return this.put(rindices, cindices, v);
    }

    public ComplexFloatMatrix putImag(int[] rindices, int[] cindices, float v) {
        for (int i = 0; i < rindices.length; ++i) {
            for (int j = 0; j < cindices.length; ++j) {
                this.put(rindices[i], cindices[j], v);
            }
        }
        return this;
    }

    public ComplexFloatMatrix put(int[] rindices, int[] cindices, ComplexFloat v) {
        for (int i = 0; i < rindices.length; ++i) {
            for (int j = 0; j < cindices.length; ++j) {
                this.put(rindices[i], cindices[j], v);
            }
        }
        return this;
    }

    public ComplexFloatMatrix put(ComplexFloatMatrix indices, ComplexFloatMatrix v) {
        return this.put(indices.findIndices(), v);
    }

    public ComplexFloatMatrix put(int r, ComplexFloatMatrix indices, ComplexFloatMatrix v) {
        return this.put(r, indices.findIndices(), v);
    }

    public ComplexFloatMatrix put(ComplexFloatMatrix indices, int c, ComplexFloatMatrix v) {
        return this.put(indices.findIndices(), c, v);
    }

    public ComplexFloatMatrix put(ComplexFloatMatrix rindices, ComplexFloatMatrix cindices, ComplexFloatMatrix v) {
        return this.put(rindices.findIndices(), cindices.findIndices(), v);
    }

    public ComplexFloatMatrix put(ComplexFloatMatrix indices, float v) {
        return this.put(indices.findIndices(), v);
    }

    public ComplexFloatMatrix putReal(ComplexFloatMatrix indices, float v) {
        return this.put(indices, v);
    }

    public ComplexFloatMatrix putImag(ComplexFloatMatrix indices, float v) {
        return this.putImag(indices.findIndices(), v);
    }

    public ComplexFloatMatrix put(ComplexFloatMatrix indices, ComplexFloat v) {
        return this.put(indices.findIndices(), v);
    }

    public ComplexFloatMatrix put(int r, ComplexFloatMatrix indices, float v) {
        return this.put(r, indices.findIndices(), v);
    }

    public ComplexFloatMatrix putReal(int r, ComplexFloatMatrix indices, float v) {
        return this.put(r, indices, v);
    }

    public ComplexFloatMatrix putImag(int r, ComplexFloatMatrix indices, float v) {
        return this.putImag(r, indices.findIndices(), v);
    }

    public ComplexFloatMatrix put(int r, ComplexFloatMatrix indices, ComplexFloat v) {
        return this.put(r, indices.findIndices(), v);
    }

    public ComplexFloatMatrix put(ComplexFloatMatrix indices, int c, float v) {
        return this.put(indices.findIndices(), c, v);
    }

    public ComplexFloatMatrix putReal(ComplexFloatMatrix indices, int c, float v) {
        return this.put(indices, c, v);
    }

    public ComplexFloatMatrix putImag(ComplexFloatMatrix indices, int c, float v) {
        return this.putImag(indices.findIndices(), c, v);
    }

    public ComplexFloatMatrix put(ComplexFloatMatrix indices, int c, ComplexFloat v) {
        return this.put(indices.findIndices(), c, v);
    }

    public ComplexFloatMatrix put(ComplexFloatMatrix rindices, ComplexFloatMatrix cindices, float v) {
        return this.put(rindices.findIndices(), cindices.findIndices(), v);
    }

    public ComplexFloatMatrix putReal(ComplexFloatMatrix rindices, ComplexFloatMatrix cindices, float v) {
        return this.putReal(rindices, cindices, v);
    }

    public ComplexFloatMatrix putImag(ComplexFloatMatrix rindices, ComplexFloatMatrix cindices, float v) {
        return this.putImag(rindices.findIndices(), cindices.findIndices(), v);
    }

    public ComplexFloatMatrix put(ComplexFloatMatrix rindices, ComplexFloatMatrix cindices, ComplexFloat v) {
        return this.put(rindices.findIndices(), cindices.findIndices(), v);
    }

    public int[] findIndices() {
        int len = 0;
        for (int i = 0; i < this.length; ++i) {
            if (this.get(i).isZero()) continue;
            ++len;
        }
        int[] indices = new int[len];
        int c = 0;
        for (int i = 0; i < this.length; ++i) {
            if (this.get(i).isZero()) continue;
            indices[c++] = i;
        }
        return indices;
    }

    public ComplexFloatMatrix transpose() {
        ComplexFloatMatrix result = new ComplexFloatMatrix(this.columns, this.rows);
        ComplexFloat c = new ComplexFloat(0.0f);
        for (int i = 0; i < this.rows; ++i) {
            for (int j = 0; j < this.columns; ++j) {
                result.put(j, i, this.get(i, j, c));
            }
        }
        return result;
    }

    public ComplexFloatMatrix hermitian() {
        ComplexFloatMatrix result = new ComplexFloatMatrix(this.columns, this.rows);
        ComplexFloat c = new ComplexFloat(0.0f);
        for (int i = 0; i < this.rows; ++i) {
            for (int j = 0; j < this.columns; ++j) {
                result.put(j, i, this.get(i, j, c).conji());
            }
        }
        return result;
    }

    public ComplexFloatMatrix conji() {
        ComplexFloat c = new ComplexFloat(0.0f);
        for (int i = 0; i < this.length; ++i) {
            this.put(i, this.get(i, c).conji());
        }
        return this;
    }

    public ComplexFloatMatrix conj() {
        return this.dup().conji();
    }

    public boolean equals(Object o) {
        if (!(o instanceof ComplexFloatMatrix)) {
            return false;
        }
        ComplexFloatMatrix other = (ComplexFloatMatrix)o;
        if (!this.sameSize(other)) {
            return false;
        }
        FloatMatrix diff = MatrixFunctions.absi(this.sub(other)).getReal();
        return (double)(diff.max() / (float)(this.rows * this.columns)) < 1.0E-6;
    }

    public void resize(int newRows, int newColumns) {
        this.rows = newRows;
        this.columns = newColumns;
        this.length = newRows * newColumns;
        this.data = new float[2 * this.rows * this.columns];
    }

    public ComplexFloatMatrix reshape(int newRows, int newColumns) {
        if (this.length != newRows * newColumns) {
            throw new IllegalArgumentException("Number of elements must not change.");
        }
        this.rows = newRows;
        this.columns = newColumns;
        return this;
    }

    public boolean sameSize(ComplexFloatMatrix a) {
        return this.rows == a.rows && this.columns == a.columns;
    }

    public void assertSameSize(ComplexFloatMatrix a) {
        if (!this.sameSize(a)) {
            throw new SizeException("Matrices must have the same size.");
        }
    }

    public boolean multipliesWith(ComplexFloatMatrix a) {
        return this.columns == a.rows;
    }

    public void assertMultipliesWith(ComplexFloatMatrix a) {
        if (!this.multipliesWith(a)) {
            throw new SizeException("Number of columns of left matrix must be equal to number of rows of right matrix.");
        }
    }

    public boolean sameLength(ComplexFloatMatrix a) {
        return this.length == a.length;
    }

    public void assertSameLength(ComplexFloatMatrix a) {
        if (!this.sameLength(a)) {
            throw new SizeException("Matrices must have same length (is: " + this.length + " and " + a.length + ")");
        }
    }

    public ComplexFloatMatrix copy(ComplexFloatMatrix a) {
        if (!this.sameSize(a)) {
            this.resize(a.rows, a.columns);
        }
        SimpleBlas.copy(a, this);
        return a;
    }

    public ComplexFloatMatrix dup() {
        ComplexFloatMatrix out = new ComplexFloatMatrix(this.rows, this.columns);
        JavaBlas.rcopy(2 * this.length, this.data, 0, 1, out.data, 0, 1);
        return out;
    }

    public ComplexFloatMatrix swapColumns(int i, int j) {
        NativeBlas.cswap(this.rows, this.data, this.index(0, i), 1, this.data, this.index(0, j), 1);
        return this;
    }

    public ComplexFloatMatrix swapRows(int i, int j) {
        NativeBlas.cswap(this.columns, this.data, this.index(i, 0), this.rows, this.data, this.index(j, 0), this.rows);
        return this;
    }

    public ComplexFloatMatrix put(int rowIndex, int columnIndex, float value) {
        this.data[2 * this.index((int)rowIndex, (int)columnIndex)] = value;
        return this;
    }

    public ComplexFloatMatrix put(int rowIndex, int columnIndex, float realValue, float complexValue) {
        this.data[2 * this.index((int)rowIndex, (int)columnIndex)] = realValue;
        this.data[2 * this.index((int)rowIndex, (int)columnIndex) + 1] = complexValue;
        return this;
    }

    public ComplexFloatMatrix put(int rowIndex, int columnIndex, ComplexFloat value) {
        int i = 2 * this.index(rowIndex, columnIndex);
        this.data[i] = value.real();
        this.data[i + 1] = value.imag();
        return this;
    }

    public ComplexFloatMatrix putReal(int rowIndex, int columnIndex, float value) {
        this.data[2 * this.index((int)rowIndex, (int)columnIndex)] = value;
        return this;
    }

    public ComplexFloatMatrix putImag(int rowIndex, int columnIndex, float value) {
        this.data[2 * this.index((int)rowIndex, (int)columnIndex) + 1] = value;
        return this;
    }

    public ComplexFloat get(int rowIndex, int columnIndex) {
        int i = 2 * this.index(rowIndex, columnIndex);
        return new ComplexFloat(this.data[i], this.data[i + 1]);
    }

    public ComplexFloat get(int rowIndex, int columnIndex, ComplexFloat result) {
        return this.get(this.index(rowIndex, columnIndex), result);
    }

    public FloatMatrix getReal() {
        FloatMatrix result = new FloatMatrix(this.rows, this.columns);
        NativeBlas.scopy(this.length, this.data, 0, 2, result.data, 0, 1);
        return result;
    }

    public int index(int rowIndex, int columnIndex) {
        return this.rows * columnIndex + rowIndex;
    }

    public ComplexFloat get(int i) {
        return new ComplexFloat(this.data[i * 2], this.data[i * 2 + 1]);
    }

    public ComplexFloat get(int i, ComplexFloat result) {
        return result.set(this.data[i * 2], this.data[i * 2 + 1]);
    }

    public float getReal(int i) {
        return this.data[2 * i];
    }

    public float getImag(int i) {
        return this.data[2 * i + 1];
    }

    public ComplexFloatMatrix put(int i, float v) {
        this.data[2 * i] = v;
        return this;
    }

    public ComplexFloatMatrix put(int i, float r, float c) {
        this.data[2 * i] = r;
        this.data[2 * i + 1] = c;
        return this;
    }

    public ComplexFloatMatrix put(int i, ComplexFloat v) {
        this.data[2 * i] = v.real();
        this.data[2 * i + 1] = v.imag();
        return this;
    }

    public ComplexFloatMatrix putReal(int i, float v) {
        return this.put(i, v);
    }

    public ComplexFloatMatrix putImag(int i, float v) {
        this.data[2 * i + 1] = v;
        return this;
    }

    public int getRows() {
        return this.rows;
    }

    public int getColumns() {
        return this.columns;
    }

    public int getLength() {
        return this.length;
    }

    public boolean isEmpty() {
        return this.columns == 0 || this.rows == 0;
    }

    public boolean isSquare() {
        return this.columns == this.rows;
    }

    public void assertSquare() {
        if (!this.isSquare()) {
            throw new SizeException("Matrix must be square!");
        }
    }

    public boolean isVector() {
        return this.columns == 1 || this.rows == 1;
    }

    public boolean isRowVector() {
        return this.columns == 1;
    }

    public boolean isColumnVector() {
        return this.rows == 1;
    }

    public ComplexFloatMatrix diag() {
        ComplexFloatMatrix d = new ComplexFloatMatrix(this.rows);
        NativeBlas.ccopy(this.rows, this.data, 0, this.rows + 1, d.data, 0, 1);
        return d;
    }

    public FloatMatrix real() {
        FloatMatrix result = new FloatMatrix(this.rows, this.columns);
        NativeBlas.scopy(this.length, this.data, 0, 2, result.data, 0, 1);
        return result;
    }

    public FloatMatrix imag() {
        FloatMatrix result = new FloatMatrix(this.rows, this.columns);
        NativeBlas.scopy(this.length, this.data, 1, 2, result.data, 0, 1);
        return result;
    }

    public void print() {
        System.out.println(this.toString());
    }

    public String toString() {
        StringBuilder s = new StringBuilder();
        s.append("[");
        for (int i = 0; i < this.rows; ++i) {
            for (int j = 0; j < this.columns; ++j) {
                s.append(this.get(i, j));
                if (j >= this.columns - 1) continue;
                s.append(", ");
            }
            if (i >= this.rows - 1) continue;
            s.append("; ");
        }
        s.append("]");
        return s.toString();
    }

    public float[] toDoubleArray() {
        float[] array = new float[2 * this.length];
        for (int i = 0; i < 2 * this.length; ++i) {
            array[i] = this.data[i];
        }
        return array;
    }

    public ComplexFloat[] toArray() {
        ComplexFloat[] array = new ComplexFloat[this.length];
        for (int i = 0; i < this.length; ++i) {
            array[i] = this.get(i);
        }
        return array;
    }

    public ComplexFloat[][] toArray2() {
        ComplexFloat[][] array = new ComplexFloat[this.rows][this.columns];
        for (int r = 0; r < this.rows; ++r) {
            for (int c = 0; c < this.columns; ++c) {
                array[r][c] = this.get(r, c);
            }
        }
        return array;
    }

    public boolean[] toBooleanArray() {
        boolean[] array = new boolean[this.length];
        for (int i = 0; i < this.length; ++i) {
            array[i] = !this.get(i).isZero();
        }
        return array;
    }

    public boolean[][] toBooleanArray2() {
        boolean[][] array = new boolean[this.rows][this.columns];
        for (int r = 0; r < this.rows; ++r) {
            for (int c = 0; c < this.columns; ++c) {
                array[r][c] = !this.get(r, c).isZero();
            }
        }
        return array;
    }

    private void ensureResultLength(ComplexFloatMatrix other, ComplexFloatMatrix result) {
        if (!this.sameLength(result)) {
            if (result == this || result == other) {
                throw new SizeException("Cannot resize result matrix because it is used in-place.");
            }
            result.resize(this.rows, this.columns);
        }
    }

    public ComplexFloatMatrix addi(ComplexFloatMatrix other, ComplexFloatMatrix result) {
        if (other.isScalar()) {
            return this.addi(other.scalar(), result);
        }
        this.assertSameLength(other);
        this.ensureResultLength(other, result);
        if (result == this) {
            SimpleBlas.axpy(ComplexFloat.UNIT, other, result);
        } else if (result == other) {
            SimpleBlas.axpy(ComplexFloat.UNIT, this, result);
        } else {
            SimpleBlas.copy(this, result);
            SimpleBlas.axpy(ComplexFloat.UNIT, other, result);
        }
        return result;
    }

    public ComplexFloatMatrix addi(ComplexFloat v, ComplexFloatMatrix result) {
        this.ensureResultLength(null, result);
        for (int i = 0; i < this.length; ++i) {
            result.put(i, this.get(i).add(v));
        }
        return result;
    }

    public ComplexFloatMatrix addi(float v, ComplexFloatMatrix result) {
        return this.addi(new ComplexFloat(v), result);
    }

    public ComplexFloatMatrix subi(ComplexFloatMatrix other, ComplexFloatMatrix result) {
        if (other.isScalar()) {
            return this.subi(other.scalar(), result);
        }
        this.assertSameLength(other);
        this.ensureResultLength(other, result);
        if (result == this) {
            SimpleBlas.axpy(ComplexFloat.NEG_UNIT, other, result);
        } else if (result == other) {
            SimpleBlas.scal(ComplexFloat.NEG_UNIT, result);
            SimpleBlas.axpy(ComplexFloat.UNIT, this, result);
        } else {
            SimpleBlas.copy(this, result);
            SimpleBlas.axpy(ComplexFloat.NEG_UNIT, other, result);
        }
        return result;
    }

    public ComplexFloatMatrix subi(ComplexFloat v, ComplexFloatMatrix result) {
        this.ensureResultLength(null, result);
        for (int i = 0; i < this.length; ++i) {
            result.put(i, this.get(i).sub(v));
        }
        return result;
    }

    public ComplexFloatMatrix subi(float v, ComplexFloatMatrix result) {
        return this.subi(new ComplexFloat(v), result);
    }

    public ComplexFloatMatrix rsubi(ComplexFloatMatrix other, ComplexFloatMatrix result) {
        return other.subi(this, result);
    }

    public ComplexFloatMatrix rsubi(ComplexFloat a, ComplexFloatMatrix result) {
        this.ensureResultLength(null, result);
        for (int i = 0; i < this.length; ++i) {
            result.put(i, a.sub(this.get(i)));
        }
        return result;
    }

    public ComplexFloatMatrix rsubi(float a, ComplexFloatMatrix result) {
        return this.rsubi(new ComplexFloat(a), result);
    }

    public ComplexFloatMatrix muli(ComplexFloatMatrix other, ComplexFloatMatrix result) {
        if (other.isScalar()) {
            return this.muli(other.scalar(), result);
        }
        this.assertSameLength(other);
        this.ensureResultLength(other, result);
        ComplexFloat c = new ComplexFloat(0.0f);
        ComplexFloat d = new ComplexFloat(0.0f);
        for (int i = 0; i < this.length; ++i) {
            result.put(i, this.get(i, c).muli(other.get(i, d)));
        }
        return result;
    }

    public ComplexFloatMatrix muli(ComplexFloat v, ComplexFloatMatrix result) {
        this.ensureResultLength(null, result);
        ComplexFloat c = new ComplexFloat(0.0f);
        for (int i = 0; i < this.length; ++i) {
            result.put(i, this.get(i, c).muli(v));
        }
        return result;
    }

    public ComplexFloatMatrix muli(float v, ComplexFloatMatrix result) {
        return this.muli(new ComplexFloat(v), result);
    }

    public ComplexFloatMatrix mmuli(ComplexFloatMatrix other, ComplexFloatMatrix result) {
        if (other.isScalar()) {
            return this.muli(other.scalar(), result);
        }
        this.assertMultipliesWith(other);
        if (result.rows != this.rows || result.columns != other.columns) {
            if (result != this && result != other) {
                result.resize(this.rows, other.columns);
            } else {
                throw new SizeException("Cannot resize result matrix because it is used in-place.");
            }
        }
        if (result == this || result == other) {
            ComplexFloatMatrix temp = new ComplexFloatMatrix(result.rows, result.columns);
            SimpleBlas.gemm(ComplexFloat.UNIT, this, other, ComplexFloat.ZERO, temp);
            SimpleBlas.copy(temp, result);
        } else {
            SimpleBlas.gemm(ComplexFloat.UNIT, this, other, ComplexFloat.ZERO, result);
        }
        return result;
    }

    public ComplexFloatMatrix mmuli(ComplexFloat v, ComplexFloatMatrix result) {
        return this.muli(v, result);
    }

    public ComplexFloatMatrix mmuli(float v, ComplexFloatMatrix result) {
        return this.muli(v, result);
    }

    public ComplexFloatMatrix divi(ComplexFloatMatrix other, ComplexFloatMatrix result) {
        if (other.isScalar()) {
            return this.divi(other.scalar(), result);
        }
        this.assertSameLength(other);
        this.ensureResultLength(other, result);
        ComplexFloat c1 = new ComplexFloat(0.0f);
        ComplexFloat c2 = new ComplexFloat(0.0f);
        for (int i = 0; i < this.length; ++i) {
            result.put(i, this.get(i, c1).divi(other.get(i, c2)));
        }
        return result;
    }

    public ComplexFloatMatrix divi(ComplexFloat a, ComplexFloatMatrix result) {
        this.ensureResultLength(null, result);
        ComplexFloat c = new ComplexFloat(0.0f);
        for (int i = 0; i < this.length; ++i) {
            result.put(i, this.get(i, c).divi(a));
        }
        return result;
    }

    public ComplexFloatMatrix divi(float a, ComplexFloatMatrix result) {
        return this.divi(new ComplexFloat(a), result);
    }

    public ComplexFloatMatrix rdivi(ComplexFloatMatrix other, ComplexFloatMatrix result) {
        if (other.isScalar()) {
            return this.divi(other.scalar(), result);
        }
        this.assertSameLength(other);
        this.ensureResultLength(other, result);
        ComplexFloat c1 = new ComplexFloat(0.0f);
        ComplexFloat c2 = new ComplexFloat(0.0f);
        for (int i = 0; i < this.length; ++i) {
            result.put(i, other.get(i, c1).divi(this.get(i, c2)));
        }
        return result;
    }

    public ComplexFloatMatrix rdivi(ComplexFloat a, ComplexFloatMatrix result) {
        this.ensureResultLength(null, result);
        ComplexFloat c1 = new ComplexFloat(0.0f);
        ComplexFloat c2 = new ComplexFloat(0.0f);
        for (int i = 0; i < this.length; ++i) {
            c1.copy(a);
            result.put(i, c1.divi(this.get(i, c2)));
        }
        return result;
    }

    public ComplexFloatMatrix rdivi(float a, ComplexFloatMatrix result) {
        return this.rdivi(new ComplexFloat(a), result);
    }

    public ComplexFloatMatrix negi() {
        ComplexFloat c = new ComplexFloat(0.0f);
        for (int i = 0; i < this.length; ++i) {
            this.put(i, this.get(i, c).negi());
        }
        return this;
    }

    public ComplexFloatMatrix neg() {
        return this.dup().negi();
    }

    public ComplexFloatMatrix noti() {
        ComplexFloat c = new ComplexFloat(0.0f);
        for (int i = 0; i < this.length; ++i) {
            this.put(i, this.get(i, c).isZero() ? 1.0f : 0.0f);
        }
        return this;
    }

    public ComplexFloatMatrix not() {
        return this.dup().noti();
    }

    public ComplexFloatMatrix truthi() {
        ComplexFloat c = new ComplexFloat(0.0f);
        for (int i = 0; i < this.length; ++i) {
            this.put(i, this.get(i, c).isZero() ? 0.0f : 1.0f);
        }
        return this;
    }

    public ComplexFloatMatrix truth() {
        return this.dup().truthi();
    }

    public ComplexFloatMatrix rankOneUpdate(ComplexFloat alpha, ComplexFloatMatrix x, ComplexFloatMatrix y) {
        if (this.rows != x.length) {
            throw new SizeException("Vector x has wrong length (" + x.length + " != " + this.rows + ").");
        }
        if (this.columns != y.length) {
            throw new SizeException("Vector y has wrong length (" + x.length + " != " + this.columns + ").");
        }
        SimpleBlas.gerc(alpha, x, y, this);
        return this;
    }

    public ComplexFloatMatrix rankOneUpdate(float alpha, ComplexFloatMatrix x, ComplexFloatMatrix y) {
        return this.rankOneUpdate(new ComplexFloat(alpha), x, y);
    }

    public ComplexFloatMatrix rankOneUpdate(float alpha, ComplexFloatMatrix x) {
        return this.rankOneUpdate(new ComplexFloat(alpha), x, x);
    }

    public ComplexFloatMatrix rankOneUpdate(ComplexFloat alpha, ComplexFloatMatrix x) {
        return this.rankOneUpdate(alpha, x, x);
    }

    public ComplexFloatMatrix rankOneUpdate(ComplexFloatMatrix x) {
        return this.rankOneUpdate(1.0f, x, x);
    }

    public ComplexFloatMatrix rankOneUpdate(ComplexFloatMatrix x, ComplexFloatMatrix y) {
        return this.rankOneUpdate(1.0f, x, y);
    }

    public ComplexFloat sum() {
        ComplexFloat s = new ComplexFloat(0.0f);
        ComplexFloat c = new ComplexFloat(0.0f);
        for (int i = 0; i < this.length; ++i) {
            s.addi(this.get(i, c));
        }
        return s;
    }

    public ComplexFloat mean() {
        return this.sum().div(this.length);
    }

    public ComplexFloat dotc(ComplexFloatMatrix other) {
        return SimpleBlas.dotc(this, other);
    }

    public ComplexFloat dotu(ComplexFloatMatrix other) {
        return SimpleBlas.dotu(this, other);
    }

    public float norm2() {
        return SimpleBlas.nrm2(this);
    }

    public float normmax() {
        int i = SimpleBlas.iamax(this);
        return this.get(i).abs();
    }

    public float norm1() {
        return SimpleBlas.asum(this);
    }

    public ComplexFloatMatrix columnSums() {
        ComplexFloatMatrix v = new ComplexFloatMatrix(1, this.columns);
        for (int c = 0; c < this.columns; ++c) {
            v.put(c, this.getColumn(c).sum());
        }
        return v;
    }

    public ComplexFloatMatrix columnMeans() {
        return this.columnSums().divi(this.rows);
    }

    public ComplexFloatMatrix rowSums() {
        ComplexFloatMatrix v = new ComplexFloatMatrix(this.rows);
        for (int r = 0; r < this.rows; ++r) {
            v.put(r, this.getRow(r).sum());
        }
        return v;
    }

    public ComplexFloatMatrix rowMeans() {
        return this.rowSums().divi(this.columns);
    }

    public ComplexFloatMatrix getColumn(int c) {
        ComplexFloatMatrix result = new ComplexFloatMatrix(this.rows, 1);
        NativeBlas.ccopy(this.rows, this.data, this.index(0, c), 1, result.data, 0, 1);
        return result;
    }

    public void putColumn(int c, ComplexFloatMatrix v) {
        NativeBlas.ccopy(this.rows, v.data, 0, 1, this.data, this.index(0, c), 1);
    }

    public ComplexFloatMatrix getRow(int r) {
        ComplexFloatMatrix result = new ComplexFloatMatrix(1, this.columns);
        NativeBlas.ccopy(this.columns, this.data, this.index(r, 0), this.rows, result.data, 0, 1);
        return result;
    }

    public void putRow(int r, ComplexFloatMatrix v) {
        NativeBlas.ccopy(this.columns, v.data, 0, 1, this.data, this.index(r, 0), this.rows);
    }

    public void addRowVector(ComplexFloatMatrix x) {
        for (int r = 0; r < this.rows; ++r) {
            NativeBlas.caxpy(this.columns, ComplexFloat.UNIT, x.data, 0, 1, this.data, this.index(r, 0), this.rows);
        }
    }

    public void addColumnVector(ComplexFloatMatrix x) {
        for (int c = 0; c < this.columns; ++c) {
            NativeBlas.caxpy(this.rows, ComplexFloat.UNIT, x.data, 0, 1, this.data, this.index(0, c), 1);
        }
    }

    public void subRowVector(ComplexFloatMatrix x) {
        for (int r = 0; r < this.rows; ++r) {
            NativeBlas.caxpy(this.columns, ComplexFloat.NEG_UNIT, x.data, 0, 1, this.data, this.index(r, 0), this.rows);
        }
    }

    public void subColumnVector(ComplexFloatMatrix x) {
        for (int c = 0; c < this.columns; ++c) {
            NativeBlas.caxpy(this.rows, ComplexFloat.NEG_UNIT, x.data, 0, 1, this.data, this.index(0, c), 1);
        }
    }

    public void out(DataOutputStream dos) throws IOException {
        dos.writeUTF("float");
        dos.writeInt(this.columns);
        dos.writeInt(this.rows);
        dos.writeInt(this.data.length);
        for (int i = 0; i < this.data.length; ++i) {
            dos.writeDouble(this.data[i]);
        }
    }

    public void in(DataInputStream dis) throws IOException {
        if (!dis.readUTF().equals("float")) {
            throw new IllegalStateException("The matrix in the specified file is not of the correct type!");
        }
        this.columns = dis.readInt();
        this.rows = dis.readInt();
        int MAX = dis.readInt();
        this.data = new float[MAX];
        for (int i = 0; i < MAX; ++i) {
            this.data[i] = dis.readFloat();
        }
    }

    public void save(String filename) throws IOException {
        DataOutputStream dos = new DataOutputStream(new FileOutputStream(filename, false));
        this.out(dos);
    }

    public void load(String filename) throws IOException {
        DataInputStream dis = new DataInputStream(new FileInputStream(filename));
        this.in(dis);
    }

    public ComplexFloatMatrix addi(ComplexFloatMatrix other) {
        return this.addi(other, this);
    }

    public ComplexFloatMatrix add(ComplexFloatMatrix other) {
        return this.addi(other, new ComplexFloatMatrix(this.rows, this.columns));
    }

    public ComplexFloatMatrix addi(ComplexFloat v) {
        return this.addi(v, this);
    }

    public ComplexFloatMatrix addi(float v) {
        return this.addi(new ComplexFloat(v), this);
    }

    public ComplexFloatMatrix add(ComplexFloat v) {
        return this.addi(v, new ComplexFloatMatrix(this.rows, this.columns));
    }

    public ComplexFloatMatrix add(float v) {
        return this.addi(new ComplexFloat(v), new ComplexFloatMatrix(this.rows, this.columns));
    }

    public ComplexFloatMatrix subi(ComplexFloatMatrix other) {
        return this.subi(other, this);
    }

    public ComplexFloatMatrix sub(ComplexFloatMatrix other) {
        return this.subi(other, new ComplexFloatMatrix(this.rows, this.columns));
    }

    public ComplexFloatMatrix subi(ComplexFloat v) {
        return this.subi(v, this);
    }

    public ComplexFloatMatrix subi(float v) {
        return this.subi(new ComplexFloat(v), this);
    }

    public ComplexFloatMatrix sub(ComplexFloat v) {
        return this.subi(v, new ComplexFloatMatrix(this.rows, this.columns));
    }

    public ComplexFloatMatrix sub(float v) {
        return this.subi(new ComplexFloat(v), new ComplexFloatMatrix(this.rows, this.columns));
    }

    public ComplexFloatMatrix rsubi(ComplexFloatMatrix other) {
        return this.rsubi(other, this);
    }

    public ComplexFloatMatrix rsub(ComplexFloatMatrix other) {
        return this.rsubi(other, new ComplexFloatMatrix(this.rows, this.columns));
    }

    public ComplexFloatMatrix rsubi(ComplexFloat v) {
        return this.rsubi(v, this);
    }

    public ComplexFloatMatrix rsubi(float v) {
        return this.rsubi(new ComplexFloat(v), this);
    }

    public ComplexFloatMatrix rsub(ComplexFloat v) {
        return this.rsubi(v, new ComplexFloatMatrix(this.rows, this.columns));
    }

    public ComplexFloatMatrix rsub(float v) {
        return this.rsubi(new ComplexFloat(v), new ComplexFloatMatrix(this.rows, this.columns));
    }

    public ComplexFloatMatrix divi(ComplexFloatMatrix other) {
        return this.divi(other, this);
    }

    public ComplexFloatMatrix div(ComplexFloatMatrix other) {
        return this.divi(other, new ComplexFloatMatrix(this.rows, this.columns));
    }

    public ComplexFloatMatrix divi(ComplexFloat v) {
        return this.divi(v, this);
    }

    public ComplexFloatMatrix divi(float v) {
        return this.divi(new ComplexFloat(v), this);
    }

    public ComplexFloatMatrix div(ComplexFloat v) {
        return this.divi(v, new ComplexFloatMatrix(this.rows, this.columns));
    }

    public ComplexFloatMatrix div(float v) {
        return this.divi(new ComplexFloat(v), new ComplexFloatMatrix(this.rows, this.columns));
    }

    public ComplexFloatMatrix rdivi(ComplexFloatMatrix other) {
        return this.rdivi(other, this);
    }

    public ComplexFloatMatrix rdiv(ComplexFloatMatrix other) {
        return this.rdivi(other, new ComplexFloatMatrix(this.rows, this.columns));
    }

    public ComplexFloatMatrix rdivi(ComplexFloat v) {
        return this.rdivi(v, this);
    }

    public ComplexFloatMatrix rdivi(float v) {
        return this.rdivi(new ComplexFloat(v), this);
    }

    public ComplexFloatMatrix rdiv(ComplexFloat v) {
        return this.rdivi(v, new ComplexFloatMatrix(this.rows, this.columns));
    }

    public ComplexFloatMatrix rdiv(float v) {
        return this.rdivi(new ComplexFloat(v), new ComplexFloatMatrix(this.rows, this.columns));
    }

    public ComplexFloatMatrix muli(ComplexFloatMatrix other) {
        return this.muli(other, this);
    }

    public ComplexFloatMatrix mul(ComplexFloatMatrix other) {
        return this.muli(other, new ComplexFloatMatrix(this.rows, this.columns));
    }

    public ComplexFloatMatrix muli(ComplexFloat v) {
        return this.muli(v, this);
    }

    public ComplexFloatMatrix muli(float v) {
        return this.muli(new ComplexFloat(v), this);
    }

    public ComplexFloatMatrix mul(ComplexFloat v) {
        return this.muli(v, new ComplexFloatMatrix(this.rows, this.columns));
    }

    public ComplexFloatMatrix mul(float v) {
        return this.muli(new ComplexFloat(v), new ComplexFloatMatrix(this.rows, this.columns));
    }

    public ComplexFloatMatrix mmuli(ComplexFloatMatrix other) {
        return this.mmuli(other, this);
    }

    public ComplexFloatMatrix mmul(ComplexFloatMatrix other) {
        return this.mmuli(other, new ComplexFloatMatrix(this.rows, other.columns));
    }

    public ComplexFloatMatrix mmuli(ComplexFloat v) {
        return this.mmuli(v, this);
    }

    public ComplexFloatMatrix mmuli(float v) {
        return this.mmuli(new ComplexFloat(v), this);
    }

    public ComplexFloatMatrix mmul(ComplexFloat v) {
        return this.mmuli(v, new ComplexFloatMatrix(this.rows, this.columns));
    }

    public ComplexFloatMatrix mmul(float v) {
        return this.mmuli(new ComplexFloat(v), new ComplexFloatMatrix(this.rows, this.columns));
    }

    public ComplexFloatMatrix eqi(ComplexFloatMatrix other, ComplexFloatMatrix result) {
        if (other.isScalar()) {
            return this.eqi(other.scalar(), result);
        }
        this.assertSameLength(other);
        this.ensureResultLength(other, result);
        ComplexFloat c1 = new ComplexFloat(0.0f);
        ComplexFloat c2 = new ComplexFloat(0.0f);
        for (int i = 0; i < this.length; ++i) {
            result.put(i, this.get(i, c1).eq(other.get(i, c2)) ? 1.0f : 0.0f);
        }
        return result;
    }

    public ComplexFloatMatrix eqi(ComplexFloatMatrix other) {
        return this.eqi(other, this);
    }

    public ComplexFloatMatrix eq(ComplexFloatMatrix other) {
        return this.eqi(other, new ComplexFloatMatrix(this.rows, this.columns));
    }

    public ComplexFloatMatrix eqi(ComplexFloat value, ComplexFloatMatrix result) {
        this.ensureResultLength(null, result);
        ComplexFloat c = new ComplexFloat(0.0f);
        for (int i = 0; i < this.length; ++i) {
            result.put(i, this.get(i, c).eq(value) ? 1.0f : 0.0f);
        }
        return result;
    }

    public ComplexFloatMatrix eqi(float value, ComplexFloatMatrix result) {
        return this.eqi(new ComplexFloat(value), result);
    }

    public ComplexFloatMatrix eqi(ComplexFloat value) {
        return this.eqi(value, this);
    }

    public ComplexFloatMatrix eqi(float value) {
        return this.eqi(new ComplexFloat(value));
    }

    public ComplexFloatMatrix eq(ComplexFloat value) {
        return this.eqi(value, new ComplexFloatMatrix(this.rows, this.columns));
    }

    public ComplexFloatMatrix eq(float value) {
        return this.eqi(new ComplexFloat(value));
    }

    public ComplexFloatMatrix nei(ComplexFloatMatrix other, ComplexFloatMatrix result) {
        if (other.isScalar()) {
            return this.nei(other.scalar(), result);
        }
        this.assertSameLength(other);
        this.ensureResultLength(other, result);
        ComplexFloat c1 = new ComplexFloat(0.0f);
        ComplexFloat c2 = new ComplexFloat(0.0f);
        for (int i = 0; i < this.length; ++i) {
            result.put(i, this.get(i, c1).eq(other.get(i, c2)) ? 1.0f : 0.0f);
        }
        return result;
    }

    public ComplexFloatMatrix nei(ComplexFloatMatrix other) {
        return this.nei(other, this);
    }

    public ComplexFloatMatrix ne(ComplexFloatMatrix other) {
        return this.nei(other, new ComplexFloatMatrix(this.rows, this.columns));
    }

    public ComplexFloatMatrix nei(ComplexFloat value, ComplexFloatMatrix result) {
        this.ensureResultLength(null, result);
        ComplexFloat c = new ComplexFloat(0.0f);
        for (int i = 0; i < this.length; ++i) {
            result.put(i, this.get(i, c).eq(value) ? 1.0f : 0.0f);
        }
        return result;
    }

    public ComplexFloatMatrix nei(float value, ComplexFloatMatrix result) {
        return this.nei(new ComplexFloat(value), result);
    }

    public ComplexFloatMatrix nei(ComplexFloat value) {
        return this.nei(value, this);
    }

    public ComplexFloatMatrix nei(float value) {
        return this.nei(new ComplexFloat(value));
    }

    public ComplexFloatMatrix ne(ComplexFloat value) {
        return this.nei(value, new ComplexFloatMatrix(this.rows, this.columns));
    }

    public ComplexFloatMatrix ne(float value) {
        return this.nei(new ComplexFloat(value));
    }

    public ComplexFloatMatrix andi(ComplexFloatMatrix other, ComplexFloatMatrix result) {
        this.assertSameLength(other);
        this.ensureResultLength(other, result);
        ComplexFloat t1 = new ComplexFloat(0.0f);
        ComplexFloat t2 = new ComplexFloat(0.0f);
        for (int i = 0; i < this.length; ++i) {
            result.put(i, !this.get(i, t1).isZero() & !other.get(i, t2).isZero() ? 1.0f : 0.0f);
        }
        return result;
    }

    public ComplexFloatMatrix andi(ComplexFloatMatrix other) {
        return this.andi(other, this);
    }

    public ComplexFloatMatrix and(ComplexFloatMatrix other) {
        return this.andi(other, new ComplexFloatMatrix(this.rows, this.columns));
    }

    public ComplexFloatMatrix andi(ComplexFloat value, ComplexFloatMatrix result) {
        this.ensureResultLength(null, result);
        boolean val = !value.isZero();
        ComplexFloat t = new ComplexFloat(0.0f);
        for (int i = 0; i < this.length; ++i) {
            result.put(i, !this.get(i, t).isZero() & val ? 1.0f : 0.0f);
        }
        return result;
    }

    public ComplexFloatMatrix andi(float value, ComplexFloatMatrix result) {
        return this.andi(new ComplexFloat(value), result);
    }

    public ComplexFloatMatrix andi(ComplexFloat value) {
        return this.andi(value, this);
    }

    public ComplexFloatMatrix andi(float value) {
        return this.andi(new ComplexFloat(value), this);
    }

    public ComplexFloatMatrix and(ComplexFloat value) {
        return this.andi(value, new ComplexFloatMatrix(this.rows, this.columns));
    }

    public ComplexFloatMatrix and(float value) {
        return this.andi(new ComplexFloat(value));
    }

    public ComplexFloatMatrix ori(ComplexFloatMatrix other, ComplexFloatMatrix result) {
        this.assertSameLength(other);
        this.ensureResultLength(other, result);
        ComplexFloat t1 = new ComplexFloat(0.0f);
        ComplexFloat t2 = new ComplexFloat(0.0f);
        for (int i = 0; i < this.length; ++i) {
            result.put(i, !this.get(i, t1).isZero() | !other.get(i, t2).isZero() ? 1.0f : 0.0f);
        }
        return result;
    }

    public ComplexFloatMatrix ori(ComplexFloatMatrix other) {
        return this.ori(other, this);
    }

    public ComplexFloatMatrix or(ComplexFloatMatrix other) {
        return this.ori(other, new ComplexFloatMatrix(this.rows, this.columns));
    }

    public ComplexFloatMatrix ori(ComplexFloat value, ComplexFloatMatrix result) {
        this.ensureResultLength(null, result);
        boolean val = !value.isZero();
        ComplexFloat t = new ComplexFloat(0.0f);
        for (int i = 0; i < this.length; ++i) {
            result.put(i, !this.get(i, t).isZero() | val ? 1.0f : 0.0f);
        }
        return result;
    }

    public ComplexFloatMatrix ori(float value, ComplexFloatMatrix result) {
        return this.ori(new ComplexFloat(value), result);
    }

    public ComplexFloatMatrix ori(ComplexFloat value) {
        return this.ori(value, this);
    }

    public ComplexFloatMatrix ori(float value) {
        return this.ori(new ComplexFloat(value), this);
    }

    public ComplexFloatMatrix or(ComplexFloat value) {
        return this.ori(value, new ComplexFloatMatrix(this.rows, this.columns));
    }

    public ComplexFloatMatrix or(float value) {
        return this.ori(new ComplexFloat(value));
    }

    public ComplexFloatMatrix xori(ComplexFloatMatrix other, ComplexFloatMatrix result) {
        this.assertSameLength(other);
        this.ensureResultLength(other, result);
        ComplexFloat t1 = new ComplexFloat(0.0f);
        ComplexFloat t2 = new ComplexFloat(0.0f);
        for (int i = 0; i < this.length; ++i) {
            result.put(i, !this.get(i, t1).isZero() ^ !other.get(i, t2).isZero() ? 1.0f : 0.0f);
        }
        return result;
    }

    public ComplexFloatMatrix xori(ComplexFloatMatrix other) {
        return this.xori(other, this);
    }

    public ComplexFloatMatrix xor(ComplexFloatMatrix other) {
        return this.xori(other, new ComplexFloatMatrix(this.rows, this.columns));
    }

    public ComplexFloatMatrix xori(ComplexFloat value, ComplexFloatMatrix result) {
        this.ensureResultLength(null, result);
        boolean val = !value.isZero();
        ComplexFloat t = new ComplexFloat(0.0f);
        for (int i = 0; i < this.length; ++i) {
            result.put(i, !this.get(i, t).isZero() ^ val ? 1.0f : 0.0f);
        }
        return result;
    }

    public ComplexFloatMatrix xori(float value, ComplexFloatMatrix result) {
        return this.xori(new ComplexFloat(value), result);
    }

    public ComplexFloatMatrix xori(ComplexFloat value) {
        return this.xori(value, this);
    }

    public ComplexFloatMatrix xori(float value) {
        return this.xori(new ComplexFloat(value), this);
    }

    public ComplexFloatMatrix xor(ComplexFloat value) {
        return this.xori(value, new ComplexFloatMatrix(this.rows, this.columns));
    }

    public ComplexFloatMatrix xor(float value) {
        return this.xori(new ComplexFloat(value));
    }
}

