/*************************************************************************
 *                                                                       *
 * Open Dynamics Engine, Copyright (C) 2001,2002 Russell L. Smith.       *
 * All rights reserved.  Email: russ@q12.org   Web: www.q12.org          *
 * Open Dynamics Engine 4J, Copyright (C) 2007-2012 Tilmann Zäschke      *
 * All rights reserved.  Email: ode4j@gmx.de   Web: www.ode4j.org        *
 *                                                                       *
 * This library is free software; you can redistribute it and/or         *
 * modify it under the terms of EITHER:                                  *
 *   (1) The GNU Lesser General Public License as published by the Free  *
 *       Software Foundation; either version 2.1 of the License, or (at  *
 *       your option) any later version. The text of the GNU Lesser      *
 *       General Public License is included with this library in the     *
 *       file LICENSE.TXT.                                               *
 *   (2) The BSD-style license that is included with this library in     *
 *       the file ODE-LICENSE-BSD.TXT and ODE4J-LICENSE-BSD.TXT.         *
 *                                                                       *
 * This library is distributed in the hope that it will be useful,       *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the files    *
 * LICENSE.TXT, ODE-LICENSE-BSD.TXT and ODE4J-LICENSE-BSD.TXT for more   *
 * details.                                                              *
 *                                                                       *
 *************************************************************************/
package org.ode4j.ode.internal.processmem;

import org.ode4j.ode.internal.Common;
import org.ode4j.ode.internal.DxBody;
import org.ode4j.ode.internal.joints.DxJoint;
import org.ode4j.ode.internal.processmem.DxUtil.BlockPointer;

public final class DxWorldProcessMemArena {

    //   public:
        //TODO        #define BUFFER_TO_ARENA_EXTRA (EFFICIENT_ALIGNMENT + dEFFICIENT_SIZE(sizeof(dxWorldProcessMemArena)))
    private final static int BUFFER_TO_ARENA_EXTRA () {
        return (DxUtil.EFFICIENT_ALIGNMENT + DxUtil.dEFFICIENT_SIZE(
                DxUtil.sizeof(DxWorldProcessMemArena.class)));
    }

    static boolean IsArenaPossible(int nBufferSize)
    {
        return DxUtil.SIZE_MAX - BUFFER_TO_ARENA_EXTRA() >= nBufferSize; // This ensures there will be no overflow
    }

    static int MakeBufferSize(int nArenaSize)
    {
        return nArenaSize - BUFFER_TO_ARENA_EXTRA();
    }

    static int MakeArenaSize(int nBufferSize)
    {
        return BUFFER_TO_ARENA_EXTRA() + nBufferSize;
    }
    //     #undef BUFFER_TO_ARENA_EXTRA

    boolean IsStructureValid() //const
    {
        return m_pAllocBegin!=null 
        && m_pAllocEnd!=null 
        && m_pAllocBegin.toInt() <= m_pAllocEnd.toInt() 
        && m_pAllocCurrent == m_pAllocBegin 
        && m_pArenaBegin!=null  
        && m_pArenaBegin.toInt() <= m_pAllocBegin.toInt(); 
    }

    int GetMemorySize() //const
    {
        return (int)m_pAllocEnd.toInt() - (int)m_pAllocBegin.toInt();
    }

    BlockPointer SaveState() //const
    {
        return m_pAllocCurrent;
    }

    void RestoreState(BlockPointer state)
    {
        m_pAllocCurrent = state;
    }

    void ResetState()
    {
        m_pAllocCurrent = m_pAllocBegin;
    }

    public BlockPointer PeekBufferRemainder() //const
    {
        return m_pAllocCurrent;
    }

    BlockPointer AllocateBlock(int size)
    {
        BlockPointer block = m_pAllocCurrent;
        m_pAllocCurrent = DxUtil.dOFFSET_EFFICIENTLY(block, size);
        Common.dIASSERT(m_pAllocCurrent.toInt() <= m_pAllocEnd.toInt());
        return block;
    }

    //          //template<typename ElementType>
    //          Class<?> AllocateArray(int count)
    //          {
    //            return (ElementType *)AllocateBlock(count * sizeof(ElementType));
    //          }
    //
    //          //template<typename ElementType>
    //          void ShrinkArray(Class<?> arr, int oldcount, int newcount)
    //          {
    //            dIASSERT(newcount <= oldcount);
    //            dIASSERT(dOFFSET_EFFICIENTLY(arr, oldcount * sizeof(ElementType)) == m_pAllocCurrent);
    //            m_pAllocCurrent = dOFFSET_EFFICIENTLY(arr, newcount * sizeof(ElementType));
    //          }

    //       public:
    //    public static void FreeMemArena (DxWorldProcessMemArena arena) {
    //        throw new UnsupportedOperationException();
    //    }
    //
    //    private static int AdjustArenaSizeForReserveRequirements(
    //            int arenareq, float rsrvfactor, int rsrvminimum) {
    //        throw new UnsupportedOperationException();
    //    }

    private BlockPointer m_pAllocBegin;
    private BlockPointer m_pAllocEnd;
    private BlockPointer m_pAllocCurrent;
    private BlockPointer m_pArenaBegin;

    DxWorldProcessMemoryManager m_pArenaMemMgr;


    //****************************************************************************
    // World processing context management

    static DxWorldProcessMemArena ReallocateMemArena (
            DxWorldProcessMemArena oldarena, int memreq, 
            final DxWorldProcessMemoryManager memmgr, double rsrvfactor, 
            int rsrvminimum)
    {
        DxWorldProcessMemArena arena = oldarena;
        boolean allocsuccess = false;

        int nOldArenaSize = 0; 
        BlockPointer pOldArenaBuffer = null;

        do {
            int oldmemsize = oldarena!=null ? oldarena.GetMemorySize() : 0;
            if (oldarena == null || oldmemsize < memreq) {
                nOldArenaSize = oldarena!=null ? MakeArenaSize(oldmemsize) : 0;
                pOldArenaBuffer = oldarena!=null ? oldarena.m_pArenaBegin : null;

                if (!IsArenaPossible(memreq)) {
                    break;
                }

                int arenareq = DxWorldProcessMemArena.MakeArenaSize(memreq);
                int arenareq_with_reserve = AdjustArenaSizeForReserveRequirements(arenareq, rsrvfactor, rsrvminimum);
                int memreq_with_reserve = memreq + (arenareq_with_reserve - arenareq);

                if (oldarena != null) {
                    oldarena.m_pArenaMemMgr.m_fnFree.run(pOldArenaBuffer, nOldArenaSize);
                    oldarena = null;

                    // Zero variables to avoid another freeing on exit
                    pOldArenaBuffer = null;
                    nOldArenaSize = 0;
                }

                // Allocate new arena
                BlockPointer pNewArenaBuffer = memmgr.m_fnAlloc.run(arenareq_with_reserve);
                if (pNewArenaBuffer == null) {
                    break;
                }

                arena = (DxWorldProcessMemArena)DxUtil.dEFFICIENT_PTR(pNewArenaBuffer).asDxWorldProcessMemArena();

                BlockPointer blockbegin = DxUtil.dEFFICIENT_PTR(arena, 1);
                BlockPointer blockend = DxUtil.dOFFSET_EFFICIENTLY(blockbegin, memreq_with_reserve);

                arena.m_pAllocBegin = blockbegin;
                arena.m_pAllocEnd = blockend;
                arena.m_pArenaBegin = pNewArenaBuffer;
                arena.m_pAllocCurrent = blockbegin;
                arena.m_pArenaMemMgr = memmgr;
            }

            allocsuccess = true;
        } while (false);

        if (!allocsuccess) {
            if (pOldArenaBuffer != null) {
                Common.dIASSERT(oldarena != null);
                oldarena.m_pArenaMemMgr.m_fnFree.run(pOldArenaBuffer, nOldArenaSize);
            }
            arena = null;
        }

        return arena;
    }

    static void FreeMemArena (DxWorldProcessMemArena arena)
    {
        int memsize = arena.GetMemorySize();
        int arenasize = DxWorldProcessMemArena.MakeArenaSize(memsize);

        BlockPointer pArenaBegin = arena.m_pArenaBegin;
        arena.m_pArenaMemMgr.m_fnFree.run(pArenaBegin, arenasize);
    }


    static int AdjustArenaSizeForReserveRequirements(int arenareq, double rsrvfactor, 
            int rsrvminimum)
    {
        double scaledarena = arenareq * rsrvfactor;
        int adjustedarena = (scaledarena < DxUtil.SIZE_MAX) ? (int)scaledarena : DxUtil.SIZE_MAX;
        int boundedarena = (adjustedarena > rsrvminimum) ? adjustedarena : rsrvminimum;
        return DxUtil.dEFFICIENT_SIZE(boundedarena);
    }

    static DxWorldProcessMemArena dxAllocateTemporaryWorldProcessMemArena(
            int memreq, final DxWorldProcessMemoryManager memmgr/*=NULL*/, 
            final DxWorldProcessMemoryReserveInfo reserveinfo/*=NULL*/)
    {
        final DxWorldProcessMemoryManager surememmgr = memmgr!=null ? memmgr : DxUtil.g_WorldProcessMallocMemoryManager;
        final DxWorldProcessMemoryReserveInfo surereserveinfo = reserveinfo!=null ? reserveinfo : DxUtil.g_WorldProcessDefaultReserveInfo;
        DxWorldProcessMemArena arena = DxWorldProcessMemArena.ReallocateMemArena(
                null, memreq, surememmgr, surereserveinfo.m_fReserveFactor, 
                surereserveinfo.m_uiReserveMinimum);
        return arena;
    }

    static void dxFreeTemporaryWorldProcessMemArena(DxWorldProcessMemArena arena)
    {
        DxWorldProcessMemArena.FreeMemArena(arena);
    }


    // ***********************************************
    // Java methods to simulate the C++ manager (TZ)
    // ***********************************************

    public final double[] AllocateArrayDReal(int size) {
        return new double[size];
    }

    public final int[] AllocateArrayInt(int size) {
        return new int[size];
    }

    /**
     * Reminder function. At the place where it is called, something needs to 
     * be implemented with respect to ProcessMemManagement.
     */
    public final static void dummy() {
        // TODO Auto-generated method stub

    }

    public final void ShrinkArrayDJointWithInfo1(
            Object[] jointiinfos,
            int _nj, int njXXX) {
        // TODO Auto-generated method stub

    }

    public final double[][] AllocateArrayDRealDReal(int n) {
        return new double[n][];
    }

    public final boolean[] AllocateArrayBool(int n) {
        return new boolean[n];
    }

    public BlockPointer BEGIN_STATE_SAVE() {
        return SaveState();
    }

    public void END_STATE_SAVE(BlockPointer saveInner) {
        RestoreState(saveInner);
    }

    public static DxWorldProcessMemArena allocateTemporary(int memreq,
            Object object, Object object2) {
        return new DxWorldProcessMemArena();
    }

    public static void freeTemporary(DxWorldProcessMemArena arena) {
        // TODO Auto-generated method stub

    }

    public DxBody[] AllocateArrayDxBody(int nb) {
        return new DxBody[nb];
    }

    public DxJoint[] AllocateArrayDxJoint(int nj) {
        return new DxJoint[nj];
    }

}
