(ns pushy.core
  (:require [goog.events :as events])
  (:import goog.History
           goog.history.Html5History
           goog.history.Html5History.TokenTransformer
           goog.history.EventType
           goog.Uri))

(defn on-click [funk]
  (events/listen js/document "click" funk))

(defn- recur-href
  "Recursively find a href value...

  This hack is because a user might click on a <span> nested in an <a> element..."
  [target]
  (if (.-href target)
    (.-href target)
    (when (.-parentNode target)
      (recur-href (.-parentNode target)))))

(defn replace-state!
  ([state]
     (replace-state! state (.-title js/document)))
  ([state title]
     (.replaceState js/history state title))
  ([state title path]
     (.replaceState js/history state title path)))

(defn scroll-state!
  "When using HTML5 pushState preserve previous scroll position

  https://gist.github.com/loganlinn/930c043331c52cb73a98"
  []
  (let [clj-state #(js->clj (.-state js/history) :keywordize-keys true)]
    (reify
      IDeref
      (-deref [_]
        (clj-state))
      IReset
      (-reset! [_ v]
        (replace-state! (clj->js v)))
      ISwap
      (-swap! [s f]
        (-reset! s (f (clj-state))))
      (-swap! [s f x]
        (-reset! s (f (clj-state) x)))
      (-swap! [s f x y]
        (-reset! s (f (clj-state) x y)))
      (-swap! [s f x y more]
        (-reset! s (apply f (clj-state) x y more))))))

(defn push-state!
  "Sets up push state for the app via goog.history.Html5History.

  Adds an event listener to all click events and dispatches `dispatch-fn`
  when the target is a href and there is a valid route.

  Takes in two functions:
  * dispatch-fn: the function that dispatches when a match is found
  * match-fn: the function used to check if the route exists
  * identity-fn: extract the route from value returned from match-fn"
  [dispatch-fn match-fn identity-fn]
  (let [transformer (TokenTransformer.)]
    (set! (.. transformer -retrieveToken)
          (fn [path-prefix location]
            (str (.-pathname location) (.-search location))))

    (set! (.. transformer -createUrl)
          (fn [token path-prefix location]
            (str path-prefix token)))

    (let [h (Html5History. js/window transformer)]
      (.setUseFragment h false)
      (.setPathPrefix h "")
      (.setEnabled h true)
      (events/listen h EventType.NAVIGATE #(dispatch-fn (identity-fn (match-fn .-token %))))
      (when-let [match (match-fn (.getToken h))]
        (dispatch-fn (identity-fn match)))
      (scroll-state!)
      (on-click
       (fn [e]
         (if-let [href (recur-href (-> e .-target))]
           (let [path (->> href  (.parse Uri) (.getPath))]
             (when (and (identity-fn (match-fn path))
                        (not (.-altKey e))
                        (not (.-ctrlKey e))
                        (not (.-metaKey e))
                        (not (.-shiftKey e)))
               (.setUseFragment h false)
               (.setPathPrefix h "")
               (.setEnabled h true)
               (. h (setToken path (-> e .-target .-title)))
               (dispatch-fn (identity-fn (match-fn path)))
               (.preventDefault e)))))))))
