(ns pushy.core
  (:require [goog.events :as events])
  (:import goog.History
           goog.history.Html5History
           goog.history.Html5History.TokenTransformer
           goog.history.EventType
           goog.Uri))

(def ^:dynamic *history* nil)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Util/private

(defn on-click [funk]
  (events/listen js/document "click" funk))

(defn- recur-href
  "Recursively find a href value

  This hack is because a user might click on a <span> nested in an <a> element"
  [target]
  (if (.-href target)
    (.-href target)
    (when (.-parentNode target)
      (recur-href (.-parentNode target)))))

(defn- replace-state!
  ([state]
     (replace-state! state (.-title js/document)))
  ([state title]
     (.replaceState js/history state title))
  ([state title path]
     (.replaceState js/history state title path)))

(defn- scroll-state!
  "When using HTML5 pushState preserve previous scroll position

  https://gist.github.com/loganlinn/930c043331c52cb73a98"
  []
  (let [clj-state #(js->clj (.-state js/history) :keywordize-keys true)]
    (reify
      IDeref
      (-deref [_]
        (clj-state))
      IReset
      (-reset! [_ v]
        (replace-state! (clj->js v)))
      ISwap
      (-swap! [s f]
        (-reset! s (f (clj-state))))
      (-swap! [s f x]
        (-reset! s (f (clj-state) x)))
      (-swap! [s f x y]
        (-reset! s (f (clj-state) x y)))
      (-swap! [s f x y more]
        (-reset! s (apply f (clj-state) x y more))))))

(defn- update-history! [h]
  (.setUseFragment h false)
  (.setPathPrefix h "")
  (.setEnabled h true)
  h)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Public API

(defn set-token!
  "Sets the history state for *history*"
  ([path]
     (. (update-history! *history*) (setToken path)))
  ([path title]
     (. (update-history! *history*) (setToken path title))))

(defn push-state!
  "Initializes push state using goog.history.Html5History

  Adds an event listener to all click events and dispatches `dispatch-fn`
  when the target element contains a href attribute that matches
  any of the routes returned by `match-fn`

  Takes in three functions:
  * dispatch-fn: the function that dispatches when a match is found
  * match-fn: the function used to check if a particular route exists
  * identity-fn: extract the route from value returned by match-fn"
  [dispatch-fn match-fn identity-fn]
  (let [transformer (TokenTransformer.)]
    (set! (.. transformer -retrieveToken)
          (fn [path-prefix location]
            (str (.-pathname location) (.-search location))))

    (set! (.. transformer -createUrl)
          (fn [token path-prefix location]
            (str path-prefix token)))

    (set! *history* (-> (Html5History. js/window transformer)
                        (update-history!)))

    ;; If the user navigates back/forward in their browser
    ;; we want to call `dispatch-fn`
    (events/listen *history* EventType.NAVIGATE
                   #(-> (.-token %) match-fn identity-fn dispatch-fn))

    ;; Initialize scroll state
    (scroll-state!)

    ;; Dispatch after initialization
    (when-let [match (match-fn (.getToken *history*))]
      (-> match identity-fn dispatch-fn))

    ;; Setup event listener on all 'click' events
    (on-click
     (fn [e]
       (when-let [href (recur-href (-> e .-target))]
         (let [path (->> href  (.parse Uri) (.getPath))]
           ;; Proceed if the identity-fn returns a value and
           ;; the user did not trigger the event via one of the
           ;; keys we should bypass
           (when (and (identity-fn (match-fn path))
                      ;; Bypass dispatch if any of these keys
                      (not (.-altKey e))
                      (not (.-ctrlKey e))
                      (not (.-metaKey e))
                      (not (.-shiftKey e))
                      ;; Bypass dispatch if middle click
                      (not= 1 (.-button e)))
             ;; Dispatch!
             (set-token! path (-> e .-target .-title))
             (.preventDefault e))))))))
