;
; Copyright (c) 2017. qlaro, Inc.
;
; Proprietary and Confidential
;
; Unauthorized copying of this project or any files within this project, via any
; medium, is strictly prohibited
;

(ns h2e.db.datomic
  (:require [clojure.java.io :as io]
            [clojure.spec.alpha :as s]
            [com.stuartsierra.component :as component]
            [datomic.api :as d]
            [io.rkn.conformity :as c]
            [taoensso.timbre :as timbre
             :refer [log trace debug info warn error fatal report
                     logf tracef debugf infof warnf errorf fatalf reportf
                     spy get-env]])
  (:import [java.io File PushbackReader]
           (datomic.peer Connection)))

;;
;; Datomic-related items
;;

(defn cassandra-uri
  [host port db-name]
  (format "datomic:cass://%s:%s/datomic.datomic/%s?user=&password=&ssl="
          host
          port
          db-name))

(defn mysql-uri
  [host port username db-name]
  (format "datomic:sql://%s?jdbc:mysql://%s:%s/datomic?user=%s"
          db-name
          host
          port
          username))

(defn psql-uri
  [{:keys [host port username password psql-db db-name]}]
  (format "datomic:sql://%s?jdbc:postgresql://%s:%s/%s?user=%s&password=%s&ssl=true"
          db-name
          host
          port
          psql-db
          username
          password))

(defn read-migration
  [^File f]
  (info "Reading migration:" f)
  (->> (io/reader f)
       (PushbackReader.)
       (clojure.edn/read {:readers *data-readers*})))

(defn get-migrations
  "Reads all available migrations from `resources/datomic/` and returns a single
  merged map that's compatible with conformity"
  [resource-dir]
  (let [migrations (-> (io/resource resource-dir)
                       (io/file)
                       (file-seq)
                       ; rest because the first is always the original directory
                       (rest))]
    (apply merge (mapv read-migration migrations))))

(defn apply-migrations!
  [conn migrations-map]
  (c/ensure-conforms conn migrations-map)
  migrations-map)

(s/def ::host string?)
(s/def ::port int?)
(s/def ::db-name string?)
(s/def ::uri string?)
(s/def ::connection (partial instance? Connection))
(s/def ::reset-db? boolean?)
(s/def ::datomic (s/keys :req-un [
                                  ::uri
                                  ::connection
                                  ::reset-db?]))

(defrecord DatomicPeer [uri connection reset-db?]
  component/Lifecycle
  (start [component]
    (info "Starting Datomic Peer")
    (let [_ (info "URI: " uri)

          _ (info "Creating database if necessary...")

          _ (when reset-db? (d/delete-database uri))


          created? (d/create-database uri)
          _ (info "Database created:" created?)

          conn (d/connect uri)
          _ (info "Connected to Datomic")

          _ (apply-migrations! conn (get-migrations "datomic/"))
          _ (info "Migrations applied")
          ]
      (merge component {:uri        uri
                        :connection conn})))

  (stop [component]
    (info "Stopping Datomic Peer")

    (when connection (d/release connection))
    (info "Released Datomic Connection")

    (merge component {:connection nil})))

(defn new-component
  [uri & {:keys [reset-db?]}]
  (map->DatomicPeer {:uri uri
                     :reset-db? reset-db?}))