;
; Copyright (c) 2017. qlaro, Inc.
;
; Proprietary and Confidential
;
; Unauthorized copying of this project or any files within this project, via any
; medium, is strictly prohibited
;

(ns h2e.db.semantic-db
  (:require [clojure.spec.alpha :as s]
            [clojurewerkz.urly.core :as urly]
            [com.stuartsierra.component :as component]
            [datomic.api :as d]
            [taoensso.timbre :as timbre
             :refer [log trace debug info warn error fatal report
                     logf tracef debugf infof warnf errorf fatalf reportf
                     spy get-env]])
  (:import (java.util UUID)
           (java.net URISyntaxException)
           (datomic.query EntityMap)
           (datomic.db Db DbId)
           (datomic.peer Connection)
           (datomic Datom)))

;;
;; general utility structures (shared between specs and functions
;;

(def type->field {:concept.type/meta-concept :instance/values
                  :concept.type/purl         :instance/purl
                  :concept.type/string       :instance/string
                  :concept.type/boolean      :instance/boolean
                  :concept.type/long         :instance/long
                  :concept.type/double       :instance/double
                  :concept.type/bigdec       :instance/bigdec
                  :concept.type/instant      :instance/instant
                  :concept.type/uuid         :instance/uuid
                  :concept.type/uri          :instance/uri
                  :concept.type/bytes        :instance/bytes})

(defn value-field
  [concept]
  (type->field (:concept/type concept)))

;;
;; specs
;;

(defn assert-spec
  [spec value]
  (when-not (s/valid? spec value)
    (let [explanation (s/explain-str spec value)]
      (throw (AssertionError. (format "(s/valid? %s %s) - %s" spec value explanation))))))

(defn assert-specs
  [& specs]
  (doseq [[spec value] specs]
    (assert-spec spec value)))

(defn valid-url?
  "Checks to make sure s is an absolute URL and has a path of more than just a
  slash"
  [^String s]
  (try
    (and (urly/absolute? s)
         (let [url (urly/url-like s)]
           (< 1 (count (urly/path-of url)))))
    (catch URISyntaxException e)))

(s/def :db/id (s/or :long #(instance? Long %)
                    :dbid-instance #(instance? DbId %)))
(s/def :concept/purl (s/and string? valid-url?))
(s/def :concept/purls (s/coll-of :concept/purl :into #{}))
(s/def :concept/type (set (keys type->field)))
(s/def :concept/cuis (s/coll-of (s/and string?) :into #{}))

(s/def :instance/of-concept (s/or :db/id :db/id
                                  :lookup-ref (s/cat :field keyword? :value :concept/purl)
                                  :entity-map (s/and #(instance? EntityMap %))
                                  ))
(s/def :instance/of-concepts (s/coll-of :instance/of-concept :into #{}))
(s/def :instance/short-display-name string?)
(s/def :instance/long-display-name string?)
; FIXME - determine a good spec for datomic ref's
(s/def :instance/values any?)
(s/def :instance/string string?)
(s/def :instance/boolean boolean?)
(s/def :instance/long number?)
(s/def :instance/double double?)
(s/def :instance/bigdec decimal?)
; FIXME - determine a good spec for datomic ref's
(s/def :instance/instant any?)
(s/def :instance/uuid uuid?)
(s/def :instance/uri uri?)
(s/def :instance/bytes bytes?)

; FIXME - determine a good spec for datomic ref's
(s/def :edge/source any?)
; FIXME - determine a good spec for datomic ref's
(s/def :edge/target any?)
(s/def :edge/relationship #{:relationship/isa})

(defn valid-instance-value?
  "Verifies that the provided instance has only a single concrete value"
  [instance]
  (let [possible-instance-types (set (vals type->field))]
    (= 1 (count (select-keys instance possible-instance-types)))))

(s/def :semantic-db/concept (s/and (s/keys :req [:concept/purls
                                                 :concept/type
                                                 :concept/preferred-name])))

(s/def :semantic-db/instance-map (s/and (s/keys :req [:instance/of-concepts]
                                                :opt [:instance/values])
                                        valid-instance-value?))
(s/def :semantic-db/instance (s/or :instance-as-map :semantic-db/instance-map
                                   :instance-as-entity (s/and #(instance? EntityMap %)
                                                              ; removing this for now since we could just pass in a valid EntityMap that hasn't been expanded
                                                              ;#(s/valid? :semantic-db/instance-map (into {} %))
                                                              )))

(s/def :datomic/entity (s/or :entity-map #(instance? EntityMap %)
                             :valid-dbid (s/keys :req [:db/id])))
;;
;; semantic db component funcs
;;

(defn semantic-db
  "Takes a system and extracts the semantic-db instance"
  [system]
  (:semantic-db system))

(defn ^:private conn
  "Given a DatomicPeer component, returns a Datomic connection for transacting"
  [datomic]
  (:connection datomic))

;;
;; definitions of SemanticDBQuery protocol and functions
;;

;
; queries (which all take a db and not a connection
;

(defprotocol SemanticDBQuery
  (concept [db purl] "Gets a concept by PURL")
  (instances [db user-uuid purl] "Gets all instances of a PURL")
  (instances-by-value [db user-uuid purl value] "Gets all instances of a PURL that have an exact match of value")
  (value [_ instance] "Retrieves the value of an instance (wrapper around fetching the concept and determining which field contains the value)")
  (expand [_ instance] "Expands an instance and nested concepts to include preferred name, descriptions, etc.")

  (related-instances [db instance relation] "Given an instance or a concept, find all other instances or concepts following the specified edge")
  )

(defn concept*
  "Takes a db instance and a string PURL and returns a Datomic entity, if
  available, null otherwise"
  [^Db db purl]
  (assert-spec :concept/purl purl)

  (d/entity db [:concept/purls purl]))

(defn instances*
  [^Db db, ^UUID user-uuid, purl]
  (assert-spec :concept/purl purl)

  (debugf "UUID type: %s, UUID: %s" (type user-uuid) user-uuid)

  (mapv #(d/entity db (first %))
        (d/q '{:find  [?instance]
               :in    [$ ?user-uuid ?purl]
               :where [[?concept :concept/purls ?purl]
                       [?user :user/id ?user-uuid]
                       [?instance :user/ref ?user]
                       [?instance :instance/of-concepts ?concept]]}
             db user-uuid purl)))

(defn value*
  "Returns the (simple) value of the instance by tracking down which field
  contains the value"
  [^Db db instance]
  (assert-spec :semantic-db/instance instance)

  ; get the corresponding concrete field of each concept
  (let [fields (set (mapv #(type->field (:concept/type %))
                          (:instance/of-concepts instance)))]
    ; while this instance can be an instance of multiple concepts, all should be
    ; using the same concrete data type or we have a knowledge management problem
    (let [num-fields (count fields)]
      (cond
        (< 1 num-fields) (throw (RuntimeException. (format "Instance '%s' has multiple concepts with conflicting types - %s" instance fields)))
        (= 0 num-fields) (throw (RuntimeException. (format "Instance '%s' appears to have no value - %s" instance fields)))))
    (get instance (first fields)))
  )

(defn expand*
  "Expands an instance and nested concepts to include preferred name, descriptions, etc."
  [db instance]
  (infof "Expanding %s" instance)
  (clojure.walk/prewalk
    (fn [x]
      (let [m (if (instance? EntityMap x)
                (into {} x)
                x)]
        (cond
          (and (coll? m) (= :instance/purl (first m))) [:instance/purl (->> (second m)
                                                                            ; only need to get the EntityMap since the prewalk
                                                                            ; will continue to expand as necessary
                                                                            (concept* db))]
          :else m)))
    instance)
  )

(defn instances-by-value*
  [^Db db, ^UUID user-uuid, purl, value]
  (assert-spec :concept/purl purl)

  (let [c (concept* db purl)
        field (value-field c)]

    (mapv #(d/entity db (first %))
          (d/q '{:find  [?instance]
                 :in    [$ ?user-uuid ?purl ?field ?value]
                 :where [[?concept :concept/purls ?purl]
                         [?user :user/id ?user-uuid]
                         [?instance :user/ref ?user]
                         [?instance :instance/of-concepts ?concept]
                         [?instance ?field ?value]]}
               db user-uuid purl field value))))

(defn related-instances*
  [^Db db instance relation]
  (assert-spec (s/merge :semantic-db/instance :datomic/entity) instance)

  (mapv #(d/entity db (first %))
        (d/q '{:find  [?e]
               :in    [$ ?source ?relation]
               :where [[?edge :edge/source ?source]
                       [?edge :edge/relationship ?relation]
                       [?edge :edge/target ?e]]}
             db (:db/id instance) relation)))

;
; updates (which all take a connection and not a db)
;

(defprotocol SemanticDBUpdate
  (transact! [_ tx])
  (insert-simple-value [_ purl value] [_ user-uuid purl value] "Creates a new instance using a simple value of a PURL.  Connects to a user if uuid specified")
  (update-simple-value [_ dbid value] "Updates instance using a simple value of existing entity.  Connects to a user if uuid specified")
  (add-relationship [_ source edge dest] "Creates a relationship SOURCE--edge-->DEST"))

(defn user-id->partition
  "Takes a user map with :user/email, :user/id (uuid) and returns a keyword
   that specifies a datomic partition"
  [user-id]
  ; if UUID, get string. otherwise, nop
  (->> (.toString user-id)
       (str "uuid-")
       (keyword "user.partition")))

(defn ->UUID
  [x]
  (if (uuid? x)
    x
    (UUID/fromString x))
  )

(defn user-uuid->partition
  "Takes a UUID or UUID string and returns a keyword that specifies a datomic
   partition"
  [user-uuid]
  (keyword "user.partition" (str "uuid-" (.toString user-uuid))))

(defn insert-simple-value*
  "Returns something immediately passable to (d/transact conn ...)"
  ([^Connection conn purl value]
   (assert-spec :concept/purl purl)

   (if-let [c (concept* (d/db conn) purl)]
     (let [type (:concept/type c)
           field (type->field type)
           instance {:instance/of-concepts #{(:db/id c)}
                     field                 value}]
       (info (format "Inserting simple value for '%s' with type '%s' in field '%s'"
                     (:concept/preferred-name c)
                     type
                     field))
       (assert-spec :semantic-db/instance instance)
       [instance])
     (throw (IllegalArgumentException. (format "Concept not found for PURL: %s" purl)))))
  ([^Connection conn user-uuid purl value]
   (assert-spec :concept/purl purl)

   (if-let [c (concept* (d/db conn) purl)]
     (let [type (:concept/type c)
           field (type->field type)
           instance {:db/id                (d/tempid (user-id->partition user-uuid))
                     :user/ref             [:user/id (->UUID user-uuid)]
                     :instance/of-concepts #{(:db/id c)}
                     field                 value}]
       (info (format "Inserting simple value for '%s' with type '%s' in field '%s'"
                     (:concept/preferred-name c)
                     type
                     field))
       (assert-spec :semantic-db/instance instance)
       [instance])
     (throw (IllegalArgumentException. (format "Concept not found for PURL: %s" purl))))))

(defn update-simple-value*
  "Returns something immediately passable to (d/transact conn ...)"
  [^Connection conn dbid value]
  (let [db (d/db conn)]
    (if-let [entity (d/entity db dbid)]
      (let [c (concept* (d/db conn) (-> entity (get-in [:instance/of-concepts]) first :concept/purls first ))
            type (:concept/type c)
            field (type->field type)
            instance {:db/id dbid
                      field  value}]
        (info (format "Updating simple value for '%s' with type '%s' in field '%s' with value '%s'"
                      dbid
                      type
                      field
                      value))
        [instance])
      (throw (IllegalArgumentException. (format "Entity not found for :db/id: %s" dbid))))))

(defn add-relationship*
  [^Connection conn source-dbid edge target-dbid]
  (assert-specs [:db/id source-dbid]
                [:db/id target-dbid])

  [{:edge/source       source-dbid
    :edge/target       target-dbid
    :edge/relationship edge}]
  )

(defn part-db
  "Given a db and a partition entity id,
  returns a view of the db with only the datoms which entities are of this partition."
  [db part]
  (debugf "Filtering partition: %s" part)
  (let [part-id (:db/id (d/entity db part))]
    (println "> PARTID: " part-id)
    (d/filter db (fn [_ ^Datom datom]
                   (println "> PART: " (-> datom .e d/part))
                   (-> datom .e d/part (= part-id))
                   true
                   ))))

(defn get-datomic-db
  [datomic-component]
  (-> datomic-component
      :connection
      d/db))

(defrecord SemanticDB [datomic]
  component/Lifecycle
  (start [this]
    (info "Starting Semantic DB")
    this)

  (stop [this]
    (info "Stopping Semantic DB")
    this)

  SemanticDBQuery
  (concept [_ purl] (concept* (d/db (:connection datomic)) purl))
  (instances [_ user-uuid purl] (instances* (get-datomic-db datomic) user-uuid purl))
  (instances-by-value [_ user-uuid purl value] (instances-by-value* (get-datomic-db datomic) user-uuid purl value))
  (value [_ instance] (value* (d/db (:connection datomic)) instance))
  (expand [_ instance] (expand* (d/db (:connection datomic)) instance))

  (related-instances [db instance relation] (related-instances* (d/db (:connection datomic)) instance relation))

  SemanticDBUpdate
  (transact! [_ tx] @(d/transact (:connection datomic) tx))
  (insert-simple-value [_ purl value] (insert-simple-value* (:connection datomic) purl value))
  (insert-simple-value [_ user-uuid purl value] (insert-simple-value* (:connection datomic) user-uuid purl value))
  (update-simple-value [_ dbid value] (update-simple-value* (:connection datomic) dbid value))
  (add-relationship [_ source edge dest] (add-relationship* (:connection datomic) source edge dest))
  )

(defn new-component
  []
  (map->SemanticDB {}))