(ns mold-http.core
  (:require [day8.re-frame.http-fx]
            [re-posh.core :as re-posh]
            [ajax.core :as ajax]
            [re-frame.core :as re-frame]
            [mold-http.model :refer [->http-event ->http-cache]]
            [mold-http.utils :as utils]))


(def noop
  (constantly nil))


(def config
  (atom nil))


(defn configure [options]
  (reset! config options))


(defn maybe-add-api-host
  "Prepends api host to requests starting with /"
  [url]
  (let [api-host (:api-host @config)]
    (if (and (some? api-host) (= \/ (first url)))
      (str api-host url)
      url)))


(defn api-request-map [{:keys [method url params success failure interceptors headers body format raw]
                        :or   {method :get
                               format :json
                               raw    false
                               body   nil}}]
  (utils/assoc-some
    {:method          method
     :uri             (maybe-add-api-host url)
     :body            body
     :format          (case format
                        :json (ajax/json-request-format)
                        :url (ajax/url-request-format)
                        nil)
     :response-format (ajax/json-response-format {:keywords? true :raw raw})
     :on-success      success
     :on-failure      failure
     :interceptors    interceptors}
    :headers headers
    :params params))


(defn add-http-event [event-name event-id]
  (let [event @(re-posh/subscribe [:http-event/pull [:http-event/name event-name]])]
    (if (some? event)
      [[:http-event/update [:http-event/name event-name] {:http-event/id event-id}]]
      [[:http-event/add (->http-event {:name event-name :id event-id})]])))


(defn add-http-cache [cache-key status response]
  (let [cache      @(re-posh/subscribe [:http-cache/pull [:http-cache/key cache-key]])
        cache-data (->http-cache {:key cache-key :status status :response response})]
    (if (some? cache)
      [[:http-cache/update [:http-cache/key cache-key] cache-data {:replace true
                                                                   :keep-nils true}]]
      [[:http-cache/add cache-data
        {:keep-nils true}]])))


(defn add-cached-response [cofx cached-response xhrio-params]
  (let [{:http-cache/keys [response status]} cached-response
        intended-cofx (get xhrio-params (utils/concat-two-keywords :on status))
        cached-cofx   [(conj intended-cofx response)]]
    (update cofx :dispatch-n concat cached-cofx)))


(defn reg-event-http
  [event-name
   request
   callback
   & {:keys [injected-cofx success-cofx failure-cofx authenticate? cache?]
      :or   {injected-cofx []
             success-cofx  []
             failure-cofx  []
             authenticate? true
             cache?        false}}]
  (let [success-key  (utils/concat-two-keywords event-name :success)
        failure-key  (utils/concat-two-keywords event-name :failure)
        complete-key (utils/concat-two-keywords event-name :complete)
        inject-cofx  [re-frame/trim-v (re-posh/inject-cofx :ds)]]

    (re-posh/reg-event-fx
      success-key
      [inject-cofx success-cofx]
      (fn [cofx [event-id cb-params result]]
        (let [new-cofx (callback cofx [nil result cb-params])]
          (cond-> (dissoc new-cofx :ds)

                  (nil? (:dispatch-n new-cofx))
                  (assoc :dispatch-n [[complete-key event-id]])

                  (not (nil? (:dispatch-n new-cofx)))
                  (update :dispatch-n conj [complete-key event-id])

                  cache?
                  (update :dispatch-n concat (add-http-cache event-id :success result))))))

    (re-posh/reg-event-fx
      failure-key
      [inject-cofx failure-cofx]
      (fn [cofx [event-id cb-params error]]
        (let [error-status       (get-in error [:parse-error :status])
              auth-failed?       (and authenticate? (= 401 error-status))
              ;; Don't call the callback fn if it's auth failure, fail silently
              new-cofx           (if auth-failed?
                                   {}
                                   (callback cofx [error nil cb-params]))
              unauthorized-event (get @config :unauthorized-event)]
          (cond-> (dissoc new-cofx :ds)

                  (nil? (:dispatch-n new-cofx))
                  (assoc :dispatch-n [[complete-key event-id]])

                  (not (nil? (:dispatch-n new-cofx)))
                  (update :dispatch-n conj [complete-key event-id])

                  ;; 401 Unauthorized
                  (and auth-failed? (some? unauthorized-event))
                  (update :dispatch-n conj unauthorized-event)))))

    (re-posh/reg-event-ds
      complete-key
      [re-frame/trim-v]
      (fn [ds [event-id]]
        (let [ident [:http-event/name event-name]]
          [[:db/retract ident :http-event/id event-id]])))

    (re-posh/reg-event-fx
      event-name
      [re-frame/trim-v injected-cofx]
      (fn [cofx [& args]]
        (when-let [request-cofx (request cofx args)]

          ;; Cached requests have to have the request id
          (when (and cache? (not (some? (get-in request-cofx [:http-xhrio :id]))))
            (js/console.warn (str "No id provided for " event-name " request. Caching will not work without a static http-xhrio id.")))

          (let [cb-params              (-> request-cofx :http-xhrio :cb-params)
                event-id               (keyword (or (get-in request-cofx [:http-xhrio :id])
                                                    (str (random-uuid))))
                dynamic-xhrio-params   {:success [success-key event-id cb-params]
                                        :failure [failure-key event-id cb-params]}
                get-auth-header        (get @config :get-auth-header noop)
                auth-header            (get-auth-header)
                xhrio-params           (cond-> (:http-xhrio request-cofx)
                                               ;; register success/failure events
                                               :always (merge dynamic-xhrio-params)
                                               ;; add auth header if some
                                               (and authenticate? (some? auth-header))
                                               (assoc-in [:headers "Authorization"] auth-header)
                                               ;; get the full request
                                               :finally api-request-map)
                transact               (add-http-event event-name event-id)
                cache-key              event-id
                cached-response        (when cache?
                                         @(re-posh/subscribe [:http-cache/pull [:http-cache/key cache-key]]))
                return-cached-version? (= (get cached-response :http-cache/status) :success)]
            (cond-> request-cofx
                    :always (dissoc :ds)
                    return-cached-version? (dissoc :http-xhrio)
                    return-cached-version? (add-cached-response cached-response xhrio-params)
                    (not return-cached-version?) (assoc :http-xhrio xhrio-params)
                    (not return-cached-version?) (update :dispatch-n concat transact))))))))


(re-frame/reg-sub
  :mold-http/loading?
  (fn [[_ event-name]]
    (re-posh/subscribe [:http-event/pull [:http-event/name event-name]]))
  (fn [http-status]
    (not (empty? (:http-event/id http-status)))))


(re-frame/reg-sub
  :mold-http/loading-id?
  (fn [[_ event-name event-id]]
    [(re-posh/subscribe [:http-event/pull [:http-event/name event-name]])
     (atom event-id)])
  (fn [[http-status event-id]]
    (some? (utils/in-seq? (:http-event/id http-status) event-id))))


(re-frame/reg-event-fx
  :mold-http/clear-caches
  (fn []
    (let [caches @(re-posh/subscribe [:http-cache/eids])]
      {:dispatch-n (mapv #(vector :http-cache/delete %) caches)})))
