(ns mold.utils
  (:require [clojure.walk :as walk]
            [spec-tools.core :as st]
            [spec-tools.transform :as stt]))


(defn remove-nils [record]
  (->> (for [[k v] record :when (nil? v)] k)
       (apply dissoc record)))


(defn remove-nils-recursive [record]
  (walk/prewalk
    (fn [node]
      (if (map? node)
        (remove-nils node)
        node))
    record))


(defn expand-keys-and-mappings [model ref-mapping]
  (fn [acc k v]
    (let [mapping-key (get ref-mapping k)]
      (assoc acc
        (keyword model (name k))
        (if (some? mapping-key)
          (cond
            (string? v)
            {mapping-key v}

            (and (vector? v) (every? string? v)) ;; TODO ref can be not only a string
            (mapv #(hash-map mapping-key %) v)

            :otherwise v)
          v)))))


(defn transform-model [spec value]
  (or
    ;; TODO would be good to get rid from the hardcoded "core.model" string
    (when (= (namespace (:name spec)) "core.model")
      (let [ref-mapping (:core.model/ref-mapping (meta spec))
            model       (name (:name spec))
            renamed     (reduce-kv
                          (expand-keys-and-mappings model ref-mapping)
                          {}
                          value)]
        renamed))
    ;; defaulting to no-op
    value))


(def model-transformer
  (st/type-transformer
    {:name            :model-transformer
     :decoders        (merge
                        stt/string-type-decoders
                        {:map transform-model})
     :default-encoder stt/any->string}))
