(ns connector.cryptog
  (:require [clj-http.client :as http]
            [clojure.data.json :as json]
            [clojure.string :as str]
            [clojure.tools.logging :as log])
  (:use [slingshot.slingshot :only [throw+ try+]])
  (:gen-class))

(defn json-reader [filename]
  (let [conf (json/read-str (slurp filename) 
                            :key-fn keyword)]
    conf))

(defn conf-reader []
  (let [conf (atom {})]
    (fn [file & args]
      (if (and (not (contains? @conf file))
               (not (nil? (first args))))
        (get (swap! conf assoc file (json-reader file))
             file)
        (get @conf file)))))

(def read-config (conf-reader))

(defn connection []
  (let [conf (read-config (str (System/getProperty "user.dir")
                               "/configuration/cryptog.json")
                          :reload)]
    conf))

(defn fetch 
  ([url-string token]
   (fetch (connection) url-string token))
  ([connection url-string token]
   (let [response (try+ (http/get (str (:url connection)
                                      url-string
                                      "/" token))
                    (catch Object e (do (log/info e)
                                         {:body nil})))]
     (when-not (or (nil? response)
                   (nil? (:body response)))
       (json/read-str (:body response)
                      :key-fn keyword)))))

(defn post 
  ([url-string params token]
   (post (connection) url-string params token))
  ([connection url-string params token]
   (let [body (json/write-str params)
         response (try+ (http/post (str (:url connection)
                                       url-string
                                       "/" token) 
                                  {:body body
                                   :content-type :json
                                   :accept :json})
                    (catch Object e (do (log/info e)
                                         nil)))]
     (when-not (nil? response)
       (json/read-str (:body response)
                      :key-fn keyword)))))


(defn settings
  [coin token]
  (let [r (fetch (str "/v1/settings/" (name coin)) token)]
    r))

(defn balance
  [coin token]
  (let [r (fetch (str "/v1/balance/" (name coin)) token)]
    r))

(defn stats
  [coin token]
  (let [r (fetch (str "/v1/user/" (name coin)) token)]
    r))

(defn nextseed
  [coin token]
  (let [r (fetch (str "/v1/nextseed/" (name coin)) token)]
    r))

(defn betinfo
  [id token]
  (let [r (fetch (str "/v1/bet/" id) token)]
    r))



(def constseed (java.util.Random. 1))

(defn randfn
  ([] (randfn (java.util.Random. 989)))
  ([r] #(format "%.0f" (* (.nextDouble r) 100000000000000))))

(def seeder (randfn))



(defn placebet
  [coin bet payout under token & seed]
  (let [client-seed (or (first seed) (seeder))]
    (let [p {:Bet (format "%.8f" (* (if (string? bet)
                                      (read-string bet)
                                      bet) 1.00000000))
                   :Payout (* (if (string? payout)
                                (read-string payout)
                                payout) 1.0000)
                   :UnderOver under
                   :ClientSeed client-seed}
          r (post (str "/v1/placebet/" (name coin))
                  p token)]
      r)))
