(ns connector.elastic
  (:require [clj-http.client :as http]
            [clojure.data.json :as json]
            [clojure.string :as str]
            )
  (:gen-class))


(defn json-reader [filename]
  (let [conf (json/read-str (slurp filename) 
                            :key-fn keyword)]
    conf))

(defn conf-reader []
  (let [conf (atom {})]
    (fn [file & args]
      (if (and (not (contains? @conf file))
               (not (nil? (first args))))
        (get (swap! conf assoc file (json-reader file))
             file)
        (get @conf file)))))

(def read-config (conf-reader))

(defn connection []
  (let [conf (read-config (str (System/getProperty "user.dir")
                                   "/configuration/elastic.json")
                              :reload)]
    conf))

(defn fetch 
  ([url-string]
   (fetch (connection) url-string))
  ([connection url-string]
   (let [_ (println (str (:host connection)
                                      url-string) "#cccccc")
         response (try (http/get (str (:host connection)
                                      url-string)
                                 {:basic-auth (:auth connection)})
                    (catch Exception e {:body nil}))]
     (when-not (or (nil? response)
                   (nil? (:body response)))
       (json/read-str (:body response)
                    :key-fn keyword)))))

(defn post 
  ([url-string params]
   (post (connection) url-string params))
  ([connection url-string params]
  (let [body (json/write-str params)
        response (try (http/post (str (:host connection)
                              url-string) 
                     {:basic-auth (:auth connection)
                      :body body
                      :content-type :json
                      :accept :json})
                   (catch Exception e (println e)))]
    (when-not (nil? response)
      (json/read-str (:body response)
                   :key-fn keyword)))))


(defn put 
  ([url-string params]
   (put (connection) url-string params))
  ([connection url-string params]
  (let [body (json/write-str params)
        response (try (http/put (str (:host connection)
                              url-string) 
                     {:basic-auth (:auth connection)
                      :body body
                      :content-type :json
                      :accept :json})
                   (catch Exception e (println e)))]
    (when-not (nil? response)
      (json/read-str (:body response)
                   :key-fn keyword)))))

(defn delete 
  ([url-string]
   (delete (connection) url-string))
  ([connection url-string]
  (let [response (try
                   (http/delete (str (:host connection)
                                  url-string)
                             {:basic-auth (:auth connection)})
                   (catch Exception e nil))]
    (when-not (nil? response)
      (json/read-str (:body response)
                   :key-fn keyword)))))

(defn head
  ([url-string]
   (head (connection) url-string))
  ([connection url-string]
   (let [response (try (http/head (str (:host connection)
                                      url-string)
                                 {:basic-auth (:auth connection)
                                  :accept :json})
                    (catch Exception e nil))]
     response)))


(defn bulk [url-path transactions]
  (when-not (nil? (first transactions))
    (let [conn (connection)
        url (str (:host conn) url-path)
        body (str/join "\n"
                       (concat (map 
                         (fn [tx]
                           (json/write-str tx)) transactions)
                               ["\n"]))
        response (try (http/post url
                            {:basic-auth (:auth conn)
                             :body body
                             :content-type :json
                             :accept :json})
                   (catch Exception e (println "POST" url body e)))]
    (when-not (nil? response)
      (json/read-str (:body response)
                   :keyword-fn keyword)))))


(defn build-uri [& args]
  (str/join "/" (map (fn [arg]
                       (if (vector? arg)
                         (str/join "," arg)
                         arg))
                     args)))


(defn update-doc
  [indx tpe id params]
  (post (build-uri indx tpe id "_update")
        {:doc params}))

(defn add-doc
  [indx tpe params]
  (post (build-uri indx tpe)
        params))

(defn bulker [& index-path]
  (let [path (apply build-uri (concat index-path ["_bulk"]))
        active-transaction (atom [])]
    (fn [& appendix]
      (if (nil? (first appendix))
        (let [transactions @active-transaction]
          (do (reset! active-transaction [])
            (bulk path transactions)))
        (swap! active-transaction concat appendix)))))




(defn exists? [& index-path]
  (let [response (apply head index-path)]
    (if (nil? response)
      false
      true)))

(defn scroll
  ([id]
   (scroll id "5m"))
  ([id scroll?]
   (let [response (post (str "_search/scroll?scroll=" scroll?)  
                  {:scroll_id id})
         results-count (count (:hits (:hits response)))]
     (lazy-seq (concat (:hits (:hits response))
                     (if-not (= 0 results-count)
                       (scroll (:_scroll_id response) scroll?)
                       [])))))
  ([previous id scroll?]
   (lazy-seq (concat previous
                     (scroll id scroll?)))))

(defn search [indexs types query & scroll?]
  (let [response (post (build-uri indexs types 
                                  (if-not (nil? (first scroll?))
                                    (str "_search?scroll=" (first scroll?))
                                    "_search"))
                        query)]
    (if (nil? (first scroll?))
      response
      (scroll (-> response :hits :hits) 
              (:_scroll_id response)
              (first scroll?)))))


(defn search-by-terms [indexs types terms & scroll?]
  (let [query {:query {:bool {:filter {:term terms}}}}]
    (search indexs types query (first scroll?))))


(defn mget 
  ([identifications]
   (mget (connection) identifications))
  ([connection identifications]
   (let [response (post "_mget" {:docs identifications})]
     response)))



(defn create-mapping 
  ([index-name mappings]
   (create-mapping (connection) index-name mappings))
  ([connection index-name mappings]
   (post connection (str index-name)
         {:mappings mappings})))



(defn distance-search 
  [indexes types coord distance maximum]
  (search indexes 
                  types 
                  {:query
                   {:bool {:must {:match_all {}}
                           :filter 
                           (merge {:geo_distance (merge coord 
                                                        {:distance distance
                                           :distance_type "arc"})})}}
                   :size maximum}
                  "5m"))
  
(defn search-terms [indexes types term ids]
  (if (< 0 (count ids))
    (search indexes
                  types
                  {:query
                   {:bool {:should (doall
                                     (map (fn [id]
                                          {:term (assoc {} term id)})
                                        ids))}}}
                  "5m")))  
  
(defn with-terms [indexes types term ids]
  (if (< 0 (count ids))
    (search indexes
                  types
                  {:query
                   {:bool {:should (doall
                                     (map (fn [id]
                                          {:term (assoc {} term id)})
                                        ids))}}}
                  )))    

  
  
(defn pull-ids [index type ids]
  (if (< 0 (count ids))
    (loop [part-ids (partition 10000 10000 nil ids)
             results []]
        (let [part-id (first part-ids)]
          (if-not (nil? part-id)
            (let [terms (reduce (fn [r id]
                                (concat r [{:_index index
                                            :_type type
                                            :_id id}]))
                                []
                                part-id)]
            (recur (rest part-ids)
                   (concat results (:docs (mget terms)))))
            results)))
    []))

