(ns kolko.cloud-backend-database.models
  (:require [clojure.spec.alpha :as s]
            [clojure.string :refer [lower-case starts-with?]]
            [clojure.set :refer [rename-keys]]))

(s/def ::id (s/and int? pos?))
(s/def ::uuid uuid?)
(s/def ::seed-campaign-id ::id)
(s/def ::platform-id (s/or :id ::id :none nil?))
(s/def ::proxy-id ::id)
(s/def ::account-id ::id)
(s/def ::ad-text-id ::id)
(s/def ::ad-text-ids (s/and (s/coll-of ::ad-text-id) seq))

(s/def ::traffic-source #{"ms-ads" "google-ads"})
(s/def ::automation-mode #{"csv" "api"})
(s/def ::platform-status #{"active" "inactive" "expired" "paused" "deleted" "local-only"
                           "budget-and-manual-paused" "budget-paused" "suspended"})
(s/def ::editorial-status #{"active" "inactive" "active-limited" "under-review" "disapproved"
                            "eligible" "pending-review" "local-only"})
(s/def ::dki boolean?)
(s/def ::keyword (s/and string? #(> (count %) 0)))
(s/def ::keywords (s/coll-of ::keyword))
(s/def ::default-keyword ::keyword)
(s/def ::force-keyword (s/or :force ::keyword :dki nil?))
(s/def ::search-term ::keyword)
(s/def ::seed-keyword ::keyword)
(s/def ::tag int?)
(s/def ::route-tag (s/or :set ::tag :unset nil?))
(s/def ::no-dki? boolean?)
;;  total return on ad spend optimizer (for Google)
(s/def ::troas? boolean?)
(s/def ::country (s/and string? #(= (count %) 2) #(= (lower-case %) %)))
(s/def ::countries (s/coll-of ::country :min-count 1))
(s/def ::language (s/and string? #(= (count %) 2) #(= (lower-case %) %)))
(s/def ::languages (s/coll-of ::language :min-count 1))
(s/def ::url-base (s/and string? #(starts-with? % "https://")))
(s/def ::system1-site (s/keys :req [::id ::url-base]))
(s/def ::remote-ad-account (s/keys :req [::id ::proxy-id ::traffic-source ::automation-mode
                                         ::account-id ::system1-site]))

(s/def ::budget (s/and float? #(>= % 0.0)))
(s/def ::bid    (s/and float? pos?))
(s/def ::cpc    (s/and float? #(>= % 0.0)))
(s/def ::volume (s/and int?   #(>= % 0)))
(s/def ::zeros  boolean?)

(s/def ::overlap int?)
(s/def ::pct-overlap float?)
(s/def ::similarity-score (s/keys :req [::id ::overlap ::pct-overlap]))

(s/def ::route (s/keys :req [::tag ::id]))

(s/def ::ad-text (s/keys :req [::title1 ::title2 ::title3 ::text1 ::text2 ::url-path ::is-default
                               ::language ::country]
                         :opt [::id]))

(s/def ::ad       (s/keys :req [::force-keyword ::platform-status ::editorial-status ::route-tag ::ad-text-id
                                ::default-keyword]

                          :opt [::id ::ad-text ::platform-id]))
(s/def ::ads      (s/coll-of ::ad))

(s/def ::adgroup-keyword  (s/keys :req [::editorial-status ::platform-status ::keyword]
                                  :opt [::platform-id ::quality-score ::cpc ::keyword-id ::adgroup-id]))
(s/def ::adgroup-keywords (s/coll-of  ::adgroup-keyword))

(s/def ::adgroup  (s/keys :req [::bid ::ads ::platform-status ::adgroup-keywords ::zeros]
                          :opt [::id ::platform-id]))
(s/def ::adgroups (s/coll-of ::adgroup))
(s/def ::campaign (s/keys :req [::traffic-source ::platform-status ::seed-campaign-id ::adgroups
                                ::countries ::languages ::budget ::remote-ad-account ::default-keyword]
                          :opt [::id ::platform-id]))

(s/def ::campaign-overview (s/keys :req [::bids ::local-only-keywords ::active-keywords ::num-adgroups
                                         ::platform-id ::platform-status ::budget
                                         ::live-tags ::live-dkis ::live-forces
                                         ::id ::created ::modified ::traffic-source
                                         ::seed-campaign-id ::seed-campaign-keyword]))

(s/def ::research-keyword  (s/keys :req [::keyword ::cpc ::volume]))
(s/def ::research-keywords (s/coll-of ::research-keyword))

(s/def ::seed-campaign (s/keys :req [::seed-keyword ::country ::research-keywords]
                               :opt [::created ::id]))

(s/def ::similarity-score  (s/keys :req [::id ::overlap ::seed-keyword ::pct-overlap]))
(s/def ::similarity-scores (s/coll-of ::similarity-score))

(defn ad-text-json->clj [ad-text]
  (rename-keys ad-text
               {:title1 ::title1
                :title2 ::title2
                :title3 ::title3
                :text1  ::text1
                :text2  ::text2
                :url_path ::url-path
                :is_default ::is-default
                :language ::language
                :country ::country
                :id ::id}))

(defn ad-text-clj->json [ad-text]
  (rename-keys ad-text
               {::title1 :title1
                ::title2 :title2
                ::title3 :title3
                ::text1 :text1
                ::text2 :text2
                ::url-path :url_path
                ::is-default :is_default
                ::language :language
                ::country :country
                ::id :id}))

;; **reporting**
(s/def ::revenue float?)
(s/def ::day-hour #(re-matches #"\d{4}-\d{2}-\d{2} \d{2}" %))
(s/def ::day-hour-data (s/keys :req [::revenue ::day-hour ::tag]))

(s/def ::campaign-id (s/or :id ::id :none nil?))
;; all the campaigns this tag has data for on this day, can be empty
(s/def ::campaigns (s/coll-of ::id :kind set?))
(s/def ::day-tz inst?)
(s/def ::total-tag-revenue ::revenue)
(s/def ::is-force boolean?)
(s/def ::total-tag-conversions int?)
(s/def ::dki-conversions int?)
(s/def ::force-conversions int?)
(s/def ::day-rev (s/keys :req [::tag ::campaign-id ::campaigns ::day-tz ::timezone ::total-tag-revenue
                               ::total-tag-conversions ::dki-conversions ::force-conversions]))
;; should have no duplicate campaign/tag
(s/def ::day-revs-normalized (s/and (s/coll-of ::day-rev)
                                    #(= (->> % (map (fn [{:keys [::tag ::campaign-id]}] [tag campaign-id]))
                                             count)
                                        (->> % (map (fn [{:keys [::tag ::campaign-id]}] [tag campaign-id]))
                                             set count))))

(def timezones #{"US/Pacific" "Europe/Berlin"})
(s/def ::timezone timezones)

;; specs for spend report metrics
(s/def ::day #(re-matches #"\d{4}-\d{2}-\d{2}" %))
(s/def ::ad-id ::id)
(s/def ::currency-code (s/and string? #(= 3 (count %))))
(s/def ::clicks int?)
(s/def ::impressions int?)
(s/def ::spend (s/and float? #(>= % 0.0)))

(s/def ::ads-spend (s/coll-of (s/keys :req [::ad-id ::clicks ::impressions ::spend ::is-force])))

(s/def ::by-type (s/coll-of (s/keys :req [::day ::is-force ::spend])))
(s/def ::by-ad   (s/coll-of (s/keys :req [::adgroup-id ::ads-spend])))

(s/def ::spend-overview (s/coll-of (s/keys :req [::id ::spend ::clicks ::impressions ::is-force])))

(s/def ::ad-day-hour (s/keys :req [::clicks ::impressions ::spend ::currency-code ::day-hour]
                             :opt [::ad-id]))

(s/def ::kw-day-hour (s/keys :req [::search-term ::clicks ::impressions ::spend ::currency-code ::day-hour]
                             :opt [::ad-id ::adgroup-id]))

;;; Mutate Operations
(s/def ::op-name #{"create", "campaign-budget", "adjust-bids", "pause", "unpause", "delete"})

;;; ** Message Queue Models ;;;
(s/def ::msg (s/or :pause!   ::pause!
                   :unpause! ::unpause!
                   :delete!  ::delete!

                   :adjust-bids! ::adjust-bids!

                   :create-local-only-campaigns! ::create-local-only-campaigns!
                   :reports-refresh-days! ::reports-refresh-days!
                   :sync-remote! ::sync-remote!))

(s/def ::entity-type #{::campaign ::adgroup ::ad})
(s/def ::pause! (s/cat :msg-name #(= % ::pause!)
                       :entity-type ::entity-type
                       :id ::id))

(s/def ::ad-to-tag (s/map-of ::id ::tag))
(s/def ::unpause! (s/cat :msg-name #(= % ::unpause!)
                         :entity-type ::entity-type
                         :id ::id
                         :ad-to-tag ::ad-to-tag))

(s/def ::delete! (s/cat :msg-name #(= % ::delete!)
                        :entity-type ::entity-type
                        :id ::id))

(s/def ::adgroup-to-bid (s/map-of ::id ::bid))
(s/def ::adjust-bids! (s/cat :msg-name #(= % ::adjust-bids!)
                             :adgroup-to-bid ::adgroup-to-bid))

(s/def ::create-local-only-campaigns! (s/cat :msg-name #(= % ::create-local-only-campaigns!)
                                             :ids (s/coll-of ::id :min-count 1)))

(s/def ::reports-refresh-days! (s/cat :msg-name #(= % ::reports-refresh-days!)))

(s/def ::sync-remote! (s/cat :msg-name #(= % ::sync-remote!)))
