(ns kosmos.sentry
  (:require [cheshire.core :as json]
            [clj-http.client :as http]
            [clojure.string :as str]
            [clojure.tools.logging :as log]
            [kosmos :refer [system]]
            [kosmos.sentry.helpers :as h]
            [kosmos.sentry.util :as u]))

(defn auth-header [ts key sig user-agent]
  (let [params [[:version   "2.0"]
                [:signature sig]
                [:timestamp ts]
                [:client    user-agent]
                [:key       key]]
        param  (fn [[k v]] (format "sentry_%s=%s" (name k) v))]
    (str "Sentry " (str/join ", " (map param params)))))

(defn exception->map [^Throwable e]
  (let [stacktraces? (not (get-in system [:sentry :ignore-stacktraces?]))
        exceptions?  (not (get-in system [:sentry :ignore-exceptions?]))]
    (cond-> {}
      stacktraces? (h/stacktrace e)
      exceptions?  (h/exception e))))

(defmulti payload-from-exception class)

(defmethod payload-from-exception clojure.lang.ExceptionInfo [e]
  (merge (exception->map e)
         {:extra (ex-data e)}))

(defmethod payload-from-exception java.lang.Throwable [e]
  (exception->map e))

(defmethod payload-from-exception :default [e]
  {})

(defn base-payload [m args exception]
  (merge
   m
   (merge-with
    merge
    (payload-from-exception exception)
    args)))

(defn event [args exception]
  (let [pid (get-in system [:sentry :pid])]
    (-> {}
        (h/server-name)
        (h/platform)
        (h/timestamp)
        (h/level)
        (h/environment)
        (h/event-id)
        (h/project pid)
        (h/sdk)
        (base-payload args exception))))

(defn report!
  "Post an event to sentry over the network."
  [{:keys [timestamp] :as event}]
  (let [dsn (get-in system [:sentry :dsn])
        ts (or timestamp (u/timestamp!))

        {:keys [key secret uri pid]} (u/parse-dsn dsn)
        sig (u/sign event ts secret)
        user-agent "kosmos-sentry"
        req {:method :post
             :url     (u/build-url uri pid)
             :headers {"X-Sentry-Auth" (auth-header ts key sig user-agent)
                       "User-Agent"    user-agent
                       "Content-Type"  "application/json"}
             :body    (json/generate-string event)}]

    (http/request req)))

;; -----
;; kosmos initialization/verification fn
;;

(defn init [{:keys [dsn environment] :as component}]
  (log/infof "initializing and validating sentry configuration with dsn=[%s] and environment=[%s]" dsn environment)
  (when-not environment
    (throw (ex-info "'environment' key must be set!" {:component component})))
  (when-not dsn
    (throw (ex-info "'dsn' key must be set!" {:component component})))
  (let [dsn-map (u/parse-dsn dsn)]
    (log/info "sentry kosmos configuration validated")
    (merge component dsn-map)))
