(ns kosmos.sentry.helpers
  (:require [digest]
            [kosmos :refer [system]]
            [kosmos.sentry.util :as u]
            [clojure.string :as str]))

;; -----
;; event map helpers
;;

(defn event-id
  ([m] (event-id m (u/uuid!)))
  ([m event-id]
   (assoc m :event_id event-id)))

(defn timestamp
  ([m] (timestamp m (u/timestamp!)))
  ([m ts]
   (assoc m :timestamp ts)))

(defn logger [m logger]
  (assoc m :logger logger))

(defn platform
  ([m] (platform m "java"))
  ([m platform]
   (assoc m :platform platform)))

(defn level
  ([m] (level m "error"))
  ([m level]
   (assoc m :level level)))

(defn culprit [m culprit]
  (assoc m :culprit culprit))

(defn server-name
  ([m] (server-name m (u/hostname)))
  ([m server-name]
   (assoc m :server_name server-name)))

(defn release [m release]
  (assoc m :release release))

(defn tags [m tags]
  (assoc m :tags tags))

(defn tag [m k v]
  (update m :tags assoc k v))


(defn get-environment []
  (if (bound? #'kosmos/system)
    (get-in system [:sentry :environment] "unknown")
    "unknown")  )

(defn environment
  ([m] (environment m (get-environment)))
  ([m environment]
   (assoc m :environment environment)))

(defn modules [m modules]
  (assoc m :modules modules))

(defn module [m module version]
  (update m :modules assoc module version))

(defn extras [m extras]
  (assoc m :extra extras))

(defn extra [m k v]
  (update m :extra assoc k v))

(defn fingerprint [m & strings]
  (assoc m :fingerprint (into ["{{ default }}"] strings)))

(defn message [m message]
  (assoc m :message message))

(defn checksum [m]
  ([m] (checksum m (digest/md5 (str (.hashCode ^Object m)))))
  ([m checksum])
  (assoc m :checksum checksum))

(defn project [m pid]
  (assoc m :project pid))

(defn sdk [m]
  (assoc m :sdk {:name "kosmos-sentry" :version (u/version "kosmos-sentry")}))

;; -----
;; stacktraces interface

(def default-not-in-app ["java" "clojure"])

(defn pattern [target]
  (if (instance? java.util.regex.Pattern target)
    target
    (re-pattern (str target))))

(defn in-app?
  "determine if the stackframe should be marked as 'in_app':

  if the classname matches the string/regex under the sequence `:inclusions` in kosmos it is marked as in_app

  otherwise
  if the classname does not match any of the string/regex under the sequence `:exclusions` it is marked as in_app
  if no exclusions are defined in kosmos the defaults of 'java' and 'clojure' are still used (ie. clojure.core would be in_app false)
  NOTE: 'java' and 'clojure' are added to exclusions"
  [classname]
  (let [exclusions (into default-not-in-app (get-in system [:sentry :exclusions] []))
        inclusions (get-in system [:sentry :inclusions] [])
        check-fn (fn [s] (re-find (pattern s) classname))]
    (boolean
     (or
      (some check-fn inclusions)
      (not-any? check-fn exclusions)))))

(defn frame->map [^StackTraceElement frame]
  (let [classname (.getClassName frame)]
    {:filename (.getFileName frame)
     :module classname
     :in_app (in-app? classname)
     :function (str
                (.getMethodName frame)
                (when (.isNativeMethod frame)
                  " (native)"))
     :lineno (.getLineNumber frame)}))

(defn stacktrace->map [^Throwable e]
  {:frames
   (for [frame (reverse (.getStackTrace e))]
     (frame->map frame))})

(defn stacktrace [m ^Throwable e]
  (assoc m :sentry.interfaces.Stacktrace (stacktrace->map e)))

(defn stacktrace-extra [m ^Throwable e]
  (let [stacktrace (u/with-err-str (.printStackTrace e))]
    (update m :extra assoc :stacktrace stacktrace)))

;; -----
;; exceptions interface
;;

(defn ex-info? [e]
  (instance? clojure.lang.ExceptionInfo e))

(defn throwable->map [^Throwable e]
  (let [_class (class e)]
    {:value     (.getMessage e)
     :module    (.getName (.getPackage _class))
     :type      (.getSimpleName _class)}))

;; ex-info exceptions all get lumped to gether which we don't want
;; so we rely on stacktrace frames only for ex-info exceptions
(defn exception [m ^Throwable e]
  (if-not (ex-info? e)
    (assoc m :sentry.interfaces.Exception  (throwable->map e))
    m))

(defn exception-extra [m ^Throwable e]
  (update m :extra assoc :exception  (format "%s: %s" (.getSimpleName (class e)) (.getMessage e))))

