(ns kosmos.sentry.util
  (:require [clj-time.core :as time]
            [clj-time.format :as fmt]
            [clojure.java.classpath :as classpath]
            [clojure.string :as str]))

(defn version [artifact]
  (let [pattern (re-pattern (str ".*/" artifact "-(.*?).jar"))
        [_ version] (->> (classpath/classpath-jarfiles)
                         (map (comp :name bean))
                         (keep (fn [jar] (re-matches pattern jar)))
                         first)]
    (or version "unknown")))

(def dsn-pattern #"^(https?)://([^:]*):([^@]*)@(.*)/([0-9]+)$")

(defn parse-dsn [dsn]
  (if-let [[_ proto key secret uri pid] (re-find dsn-pattern dsn)]
    {:key    key
     :secret secret
     :uri    (format "%s://%s" proto uri)
     :pid    (Long. ^String pid)}
    (throw (ex-info "could not parse sentry DSN" {:dsn dsn}))))

(defn random-uuid []
  (java.util.UUID/randomUUID))

(defn uuid! []
  (str/replace (str (random-uuid)) "-" ""))

(defn hostname []
  (.getHostName (java.net.InetAddress/getLocalHost)))

(defn timestamp! []
  (let [iso8601 (fmt/with-zone
                  (fmt/formatter "YYYY-MM-dd'T'HH:mm:ss")
                  (org.joda.time.DateTimeZone/forTimeZone (java.util.TimeZone/getTimeZone "UTC")))]
    (fmt/unparse iso8601 (time/now))))

(defn sign [payload ts secret]
  (let [key (javax.crypto.spec.SecretKeySpec. (.getBytes ^String secret) "HmacSHA1")
        bs  (-> (doto (javax.crypto.Mac/getInstance "HmacSHA1")
                  (.init key))
                (.doFinal (.getBytes (format "%s %s" ts payload))))]
    (String.
     (org.apache.commons.codec.binary.Base64/encodeBase64
      bs)
     "UTF-8")))

(defn build-url [uri pid]
  (str uri "/api/" pid "/store/"))

(defmacro with-err-str [& body]
  `(let [s# (new java.io.StringWriter)]
     (binding [*err* s#]
       ~@body
       (str s#))))
