(ns late.system
  (:require [clojure.java.io :as io] [clojure.string :as str])
)

;----------------------------------------------------------------------------

(defprotocol Close 
 (close [th] ))

;--------------------------------------------------------------------------- 

(defrecord Proc [cmd env dir proc 
	in-stream in-writer
	out-seq out-reader out-stream 
	err-seq err-reader err-stream
	exit]
  Close
    (close [th]	(.destroy proc))
)

;----------------------------------------------------------------------------

(defn process
  "command - string or string sequence like \"echo AA\" or [\"echo\" \"AA\"].
   Note: for bash scripts use [\"bash\" \"-c\" \"echo AA | cat\"]
   env - environment map like {\"LANG\" \"POSIX\"}
   dir - work dir for running within
   Returns Proc record. 
  "
  [command & {:keys [env dir]}]
  (let [
     _cmd (cond 
       (sequential? command) (into-array (map str command))
       (string? command) command
       :else (throw (ex-info "Bad parameter: command must be a string or seq" {:command command})))
     _env (cond (sequential? env) (into-array env) :else nil)
     _dir (cond 
        (nil? dir) nil
        (= (class dir) java.io.File) dir
        :else (io/as-file dir))
      _proc (.. (Runtime/getRuntime) (exec _cmd _env _dir))

      _in-stream (.getOutputStream _proc)
      _in-writer (io/writer _in-stream)

      _out-stream (.getInputStream _proc)
      _out-reader (io/reader _out-stream)

      _err-stream (.getErrorStream _proc)
      _err-reader (io/reader _err-stream)
      
      p ( map->Proc { 	:cmd _cmd
                	:env _env
                	:dir _dir
                	:proc _proc
                	
                	:in-writer _in-writer
                	:in-stream _in-stream
                	
                	:out-seq (lazy-seq (line-seq _out-reader))
                	:out-reader _out-reader
                	:out-stream _out-stream
                	
                	:err-seq (lazy-seq (line-seq _err-reader))
                	:err-reader _err-reader
                	:err-stream _err-stream
                	
                	:exit (future (.waitFor _proc)) 
                	}
                	)
    ]
    p
  )
)

;-------------------------------------------------------------------------------

(defn shell
  "Wrapper for \"process\" function.
  Sample: (shell [\"bash\" \"-c\" \"bash-command\"] :return :out-seq)
  command-coll - collection of commands
  opt - key-value pairs:
    key :return
	values 	:process - returns Proc record
		:out-seq - returns output lines sequence
		by default - returns exit status with printing all to *out*
  "

  [command-coll & opt]
  (let [
      p (apply process command-coll opt)
      {:keys [return]} opt
    ]
    (case return
      :process p
      :out-seq (do
        (future (binding [*out* *err*] (doseq [l (:err-seq p)] (println l))))
        (:out-seq p))
      (do ;default case:
        (future (binding [*out* *err*] (doseq [l (:err-seq p)] (println l))))
        (future (doseq [l (:out-seq p)] (println l)))
        (deref (:exit p))
      )    
    )
  )
)

;------------------------------------------------------------------------------

(defn bash-c
 "Wrapper for \"shell\" function.
 Sample: (bash-c \"for i in aa bb cc;do echo $i; sleep 1; done\" :return :out-seq)
 bash-cmd-str - bash command as string
 opt - key-value pairs:
    key :return 
	value :out-seq - returns seq of output lines
	value :process - return Proc record
	by default - returns exit status with printing all to *out*
 \""

 [bash-cmd-str & opt]
 (apply shell ["bash" "-c" bash-cmd-str] opt)
)

;------------------------------------------------------------------

(defn split-by-str [^String s ^String c]
 (loop [rv (transient []) i0 0]
   (let [i (.indexOf s c i0)]
     (if (< i 0)
       (persistent! (conj! rv (.substring s i0 (count s))))
       (recur (conj! rv (subs s i0 i)) (inc i) )
     )  
   )
 )
)


