(ns late.system
  (:require [clojure.java.io :as io] [clojure.string :as str])
)

;-------------------------------------------------------------------------
(defn line-seq-map-filter
  "1 arguments: 'rdr' - like line-seq, but first line is lazy too.
   2 arguments: 'rdr' and 'map'-predicate - like as (map predicate (line-seq rdr)), but more performance
   3 arguments: 'rdr', 'map'-predicate and 'filter'-predicate. - like as (filter fpred (map mpred (line-seq rdr)))
   Note. Will close rdr after read all lines."
  {:static true}
  ([^java.io.BufferedReader rdr] 
    (lazy-seq 
      (if-let [line (.readLine rdr)]
        (cons line (lazy-seq (line-seq-map-filter rdr)))
        (.close rdr))))
  ([^java.io.BufferedReader rdr mpred]
    (lazy-seq 
      (if-let [line (.readLine rdr)]
          (cons (mpred line) (lazy-seq (line-seq-map-filter rdr mpred)))
        (.close rdr))))
  ([^java.io.BufferedReader rdr mpred fpred]
    (lazy-seq 
      (if-let [line (.readLine rdr)]
        (let [ml (mpred line)]
          (if (fpred ml)
            (cons ml (lazy-seq (line-seq-map-filter rdr mpred fpred)))
                     (lazy-seq (line-seq-map-filter rdr mpred fpred))))
        (.close rdr)))))




;----------------------------------------------------------------------------

(defprotocol Close 
 (close [th] ))

(defprotocol SysProc
 (out-seq [th] [th opt])
 (err-seq [th]))

;--------------------------------------------------------------------------- 

(defrecord Proc [cmd env dir proc 
	in-stream in-writer
	out-reader out-stream 
	err-reader err-stream
	exit]
  Close
    (close [th]	(.destroy proc))
  SysProc
    (out-seq [th]
	(line-seq-map-filter out-reader))
    (out-seq [th opt] (let [ _map (let [m (:map opt)] (cond
                                  (fn? m) m
                                  (nil? m) nil
                                  :else (throw (ex-info "Bad map predicate." {:got m :must-be-one-of ["function","nil"]}))))

     
    		_filter (let [g (:filter opt)] (cond 
                                   (fn? g) g 
                                   (nil? g) (if (some? _map) (fn[_] true) nil)
                                   :else (throw (ex-info "Bad filter predicate." {:got g :must-be-one-of ["function","nil"]})))) ]
                (cond 
                	(some? _filter) (line-seq-map-filter out-reader _map _filter)
                	(some? _map) (line-seq-map-filter out-reader _map)
                	:else (line-seq-map-filter out-reader))))
    (err-seq [th] (lazy-seq (line-seq err-reader))))

;----------------------------------------------------------------------------

(defn process
  "Executes system command.
   command - string or string sequence like \"echo AA\" or [\"echo\" \"AA\"].
   Note: for bash scripts use [\"bash\" \"-c\" \"echo AA | cat\"]
   env - environment map like {\"LANG\" \"POSIX\"}. Optional.
   dir - work dir for running within. Optional.
   Returns Proc record. 
   
   Notes:
   It may be (and need to be) \"close\"d. 
   It has :exit field as future. 
   Attempt to dereference value in :exit field will blocks current thread until process exit.
   See (source late.system/->Proc) for other fields.
   
   Samples:
    (process \"ls\");
    (with-open [p (process [\"bash\" \"-c\" \"for i in 10 20 30; do sleep 2; echo $i; done \"] )]
    	(doseq [l (out-seq p)] (println l))) ; -print line by line
  \"\";"
  

  [command & {:keys [env dir] :as opt}]
  (let [
     _cmd (cond 
       (sequential? command) (into-array (map str command))
       (string? command) command
       :else (throw (ex-info "Bad parameter: command must be a string or seq" {:command command})))
     _env (cond (sequential? env) (into-array env) :else nil)
     _dir (cond 
        (nil? dir) nil
        (= (class dir) java.io.File) dir
        :else (io/as-file dir))
      _proc (.. (Runtime/getRuntime) (exec _cmd _env _dir))

      _in-stream (.getOutputStream _proc)
      _in-writer (io/writer _in-stream)

      _out-stream (.getInputStream _proc)
      _out-reader (io/reader _out-stream)

      _err-stream (.getErrorStream _proc)
      _err-reader (io/reader _err-stream)

      
      
      p ( map->Proc { 	:cmd _cmd
                	:env _env
                	:dir _dir
                	:proc _proc
                	
                	:in-writer _in-writer
                	:in-stream _in-stream
                	
                	:out-reader _out-reader
                	:out-stream _out-stream
                	
                	:err-reader _err-reader
                	:err-stream _err-stream
                	
                	:exit (future (.waitFor _proc)) 
                	}
                	)
    ]
    p
  )
)

;-------------------------------------------------------------------------------

(defn shell
  "Wrapper for \"process\" function.
  Sample: (shell [\"bash\" \"-c\" \"bash-command\"] :return :out-seq)
  command-coll - collection of commands
  opt - key-value pairs:
    key :return
	values 	:process - returns Proc record
		:out-seq - returns output lines sequence
		by default - returns exit status with printing all to *out*
  "

  [command-coll & opt]
  (let [
      {:keys [return]} opt
      p (apply process command-coll opt)
    ]
    (case return
      :process p
      :out-seq (do
        (future (binding [*out* *err*] (doseq [l (.err-seq p)] (println l))))
        (.out-seq p)
        )
      nil (do ;default case:
        (future (binding [*out* *err*] (doseq [l (.err-seq p)] (println l))))
        (future (doseq [l (.out-seq p)] (println l)))
        (deref (:exit p))
        )
      (throw (ex-info "Bad argument for :return" 
        { :got return 
    	  :must-be-one-of [:process :out-seq nil] } )) 
    )
  )
)

;------------------------------------------------------------------------------

(defn bash-c
 "Wrapper for \"shell\" function.
 Sample: (bash-c \"for i in aa bb cc;do echo $i; sleep 1; done\" :return :out-seq)
 bash-cmd-str - bash command as string
 opt - key-value pairs:
    key :return 
	value :out-seq - returns seq of output lines
	value :process - return Proc record
	by default - returns exit status with printing all to *out*
 \""

 [bash-cmd-str & opt]
 (apply shell ["bash" "-c" bash-cmd-str] opt)
)

;------------------------------------------------------------------
;
;(defn split-by-str-experiment-with-tranzients [^String s ^String c]
;  (loop [rv (transient []) i0 0]
;    (let [i (.indexOf s c i0)]
;      (if (< i 0)
;        (persistent! (conj! rv (.substring s i0 (count s))))
;        (recur (conj! rv (subs s i0 i)) (inc i) )
;        )
;    )
;  )
;)


