(ns diff)

(defn parse-chunk-header
  [chunk-header]
  (map #(Integer/parseInt %) (rest (re-find #"\-(\d+).*?\+(\d+)" chunk-header))))

(def chunk-re #"@@ .+? @@\s")

(defn line-type
  [line]
  (cond
   (.startsWith line "-") :delete
   (.startsWith line "+") :add
   :else :context))

(defn line-numbers
  ([lines start-left start-right]
     (line-numbers lines [] [] start-left start-right))
  ([[line & tail] lefts rights left-i right-i]
     (if line
       (let [type (line-type line)
             lefts (conj lefts (if-not (= type :add) left-i nil))
             rights (conj rights (if-not (= type :delete) right-i nil))
             left-i (if-not (= type :add)
                      (inc left-i)
                      left-i)
             right-i (if-not (= type :delete)
                       (inc right-i)
                       right-i)]
         (recur tail lefts rights left-i right-i))
       [lefts rights])))

(defn no-newline-at-end?
  [s]
  (= s "\\ No newline at end of file"))

(defn parse-lines
  [s]
  (->> (clojure.string/split s chunk-re)
       (rest)
       (map clojure.string/split-lines)
       (map #(filter (complement no-newline-at-end?) %))))

(defn parse-diff
  [s]
  (let [headers (re-seq chunk-re s)
        contents (parse-lines s)
        combined (interleave headers contents)]
    (map (fn [[header lines]]
           (conj (apply line-numbers lines (parse-chunk-header header))
                 lines))
         (partition 2 combined))))

(comment
  (def sample1 "@@ -391,7 +391,9 @@ class User < ActiveRecord::Base
   end

   def preferred_time_zone
-    timezone || account.try(:time_zone)
+    zone = timezone || account.try(:time_zone)
+    return 'Kyiv' if zone == 'Kyev'
+    zone
   end

   def last_commit_at(repository)
")

  (def sample2 "@@ -1,5 +1,5 @@
 blah
-
+ a
 blah2

 sd
@@ -9,4 +9,4 @@

 blah3

-dsfa
+usfa
"))
