(ns subversion.internals
  (:use
   fastbeans.utils
   simplelog.use)
  (:require
   [fastbeans.rpc :as rpc]
   [subversion-clj.core :as core]
   [subversion-clj.local :as local]
   [subversion-clj.wc :as wc]
   [subversion-clj.utils :as utils]
   [clojure.core.memoize :as memo])
  (:import
   [org.tmatesoft.svn.core SVNAuthenticationException]))

(defn- get-repo
  [repo-cred]
  (apply core/repo-for [(repo-cred "url") (repo-cred "login") (repo-cred "password")]))

(def cached-get-repo (memo/memo-lru get-repo 100))

(defn client-manager
  [repo-cred]
  (wc/client-manager (repo-cred "login") (repo-cred "password")))

(defn structured-diff-for
  [repo revision]
  (let [generator (local/structured-generator)]
    (local/diff-for! repo revision generator)))

(defmacro callfn
  [name args & body]
  `(defn ~name
     ~(into [] (concat ['repo-cred] args))
     (try
       (let [~'repo (cached-get-repo ~'repo-cred)]
         ~@body)
       (catch SVNAuthenticationException e#
           (rpc/die :authentication-error (str e#))))))

(defn as-int
  [x]
  (if (integer? x)
    x
    (Integer/parseInt x)))

(defn revision-info
  [log-obj]
  (let [message (.getMessage log-obj)]
    {:revision (.getRevision log-obj)
     :author (.getAuthor log-obj)
     :time (.getDate log-obj)
     :message (if message (clojure.string/trim message) "")}))

(defn cheap-revision-for
  [repo revision]
  (let [revision (long revision)]
    (->> (.log repo (utils/string-array) (utils/linked-list) revision revision true false)
      first
      revision-info)))

(defn cheap-revisions-for
  [repo rev-a rev-b]
  (->> (.log repo (utils/string-array) (utils/linked-list) (long rev-a) (long rev-b) true false)
      (map revision-info)
      (into [])))
