(ns duck-repled.definition-resolvers
  (:require [clojure.string :as str]
            [duck-repled.connect :as connect]
            [com.wsscode.pathom3.connect.operation :as pco]
            [orbit.evaluation :as eval]
            [duck-repled.template :refer [template]]
            [duck-repled.editor-helpers :as helpers]
            [promesa.core :as p]
            #?(:cljs ["fs" :refer [existsSync statSync]])
            #?(:cljs ["path" :refer [join]])
            #?(:cljs ["os" :refer [platform]]))
  #?(:clj (:import [java.io File]
                   [java.nio.file FileSystems])))

(defn- join-paths [paths]
  #?(:cljs (apply join paths)
     :clj (if (empty? paths)
            "."
            (let [[fst & rst] paths]
              (-> (FileSystems/getDefault)
                  (.getPath fst (into-array rst))
                  str)))))

(defn- file-exists? [path]
  #?(:clj (.isFile (File. path))
     :cljs (and (existsSync path)
                (.isFile (statSync path)))))

(connect/defresolver join-paths-resolver [{:keys [:file/path]}]
  {:file/filename (join-paths path)})

(connect/defresolver file-exists-resolver [{:keys [:file/filename]}]
  {:file/exists? (file-exists? filename)})

(connect/defresolver position-resolver [{:keys [:var/meta]}]
  {::pco/output [:definition/row :definition/col]}

  (when-let [line (:line meta)]
    (cond-> {:definition/row (-> meta :line dec)}
            (:column meta) (assoc :definition/col (-> meta :column dec)))))

(defn- norm-result [file-name]
  (let [os #?(:clj (System/getProperty "os.name")
              :cljs (platform))]
    (cond-> file-name
            (and (re-find #"(?i)^win" os))
            (str/replace-first #"^/" ""))))

(connect/defresolver seed-filename [input]
  {::pco/input [(pco/? :var/meta) (pco/? :ex/filename)]
   ::pco/output [:file/filename]
   ::pco/priority 30}
  (when-let [file (or (-> input :ex/filename) (-> input :var/meta :file))]
    {:file/filename file}))

(connect/defresolver existing-row-col [inputs]
  {::pco/input [(pco/? :ex/row) (pco/? :ex/col)]
   ::pco/output [:definition/row :definition/col]
   ::pco/priority 30}
  (cond-> {}
    (contains? inputs :ex/row) (assoc :definition/row (-> (:ex/row inputs) (or 1) dec))
    (contains? inputs :ex/col) (assoc :definition/col (-> (:ex/col inputs) (or 1) dec))))

(connect/defresolver existing-filename [{:keys [:file/filename :file/exists?]}]
  {::pco/input [:file/filename :file/exists?]
   ::pco/output [:definition/filename :definition/contents]
   ::pco/priority 30}
  (when exists?
    {:definition/filename filename :definition/contents nil}))

(defn- read-jar [repl jar-file-name]
  (let [[jar path] (str/split jar-file-name #"!/" 2)
        jar (clojure.string/replace-first jar #"file:" "")
        t `(let [jar-file# (java.util.jar.JarFile. ::jar)
                 ba# (java.io.ByteArrayOutputStream.)
                 is# (.getInputStream jar-file# (.getJarEntry jar-file# ::path))]
             (clojure.java.io/copy is# ba#)
             (java.lang.String. (.toByteArray ba#)))
        code (template t {::jar jar ::path path})]
    (eval/evaluate repl code {:options {:kind :clj/aux}})))

(connect/defresolver clojure-filename [{:keys [:repl/evaluator :var/meta :repl/kind]}]
  {::pco/output [:definition/filename
                 {:definition/contents [:text/contents :text/range]}]}

  (p/let [code (template `(do
                            (require 'clojure.java.io)
                            (some->> :file
                                     (.getResource (clojure.lang.RT/baseLoader))
                                     .getPath))
                         (select-keys meta [:file]))
          {:keys [result]} (eval/evaluate evaluator code {:options {:kind :clj/aux}})
          filename (norm-result result)]
    (if (re-find #"\.jar!/" filename)
      (p/let [{:keys [result]} (read-jar evaluator filename)
              pos [(-> meta (:line 1) dec) (-> meta (:column 1) dec)]]
        {:definition/filename filename
          :definition/contents {:text/contents result
                                :text/range [pos pos]}})
      {:definition/filename filename})))

(connect/defresolver file-from-clr [{:keys [:repl/evaluator :var/meta :repl/kind]}]
  {::pco/output [:definition/filename]}
  #_
  (when (= :cljr kind)
    (p/let [code (template `(some-> ::file
                                    clojure.lang.RT/FindFile
                                    str)
                           {::file (:file meta)})
            {:keys [result]} (repl/eval evaluator code)]
      (when result
        {:definition/filename (norm-result result)}))))

(defn- extract-right-var [current-var contents]
  (let [contents (or (:text/contents current-var) contents)
        [_ var] (helpers/current-var (str contents) [0 0])]
    (when (and var (= var contents))
      contents)))

(connect/defresolver resolver-for-ns-only
  [{:keys [repl/evaluator text/contents text/current-var]}]
  {::pco/input [:repl/evaluator (pco/? :text/current-var) (pco/? :text/contents)]
   ::pco/output [:var/meta :definition/row :definition/col]}

  (when-let [fqn (some-> (extract-right-var current-var contents) symbol)]
    (when (-> fqn namespace nil?)
      (p/let [code (template `(let [ns# (find-ns '::namespace-sym)
                                    first-var# (some-> ns#
                                                       ns-interns
                                                       first
                                                       second
                                                       meta
                                                       :file)
                                     ns-meta# (meta ns#)]
                                (cond-> ns-meta#
                                        first-var# (assoc :file first-var#)))
                             {::namespace-sym fqn})
              {:keys [result]} (eval/evaluate evaluator code {:options {:kind :clj/aux}})
              col (-> current-var :text/range first second)]
        (when result
          (cond-> {:var/meta result
                   ;; FIXME - this is DEFINITELY not right
                   :definition/row (dec (:line result 1))}
                  col (assoc :definition/col col)))))))

(connect/defresolver resolver-for-stacktrace [{:repl/keys [evaluator]
                                               :ex/keys [function-name filename]}]
  {::pco/input [:repl/evaluator :ex/function-name :ex/filename]
   ::pco/output [:var/meta]
   ::pco/priority 20}

  (p/let [ns-name (-> function-name (str/split #"/") first)
          code (template `(let [n# (find-ns '::namespace-sym)]
                            (->> n#
                                 ns-interns
                                 (some (fn [[_# res#]]
                                         (let [meta# (meta res#)
                                               file# (-> meta# :file str)]
                                           (and (clojure.string/ends-with? file#
                                                                           ::file-name)
                                                (select-keys meta# [:file])))))))
                         {::namespace-sym (symbol ns-name)
                          ::file-name filename})
          {:keys [result]} (eval/evaluate evaluator code {:options {:kind :clj/aux}})]
    {:var/meta result}))

(connect/defresolver source-from-contents [inputs]
  {::pco/input [{:definition/contents [:text/top-block]}]
   ::pco/output [:definition/source]
   ::pco/priority 1}

  (when-let [source (-> inputs :definition/contents :text/top-block)]
   {:definition/source source}))

;; FIXME: Maybe this will not cache top-blocks from the same filename
(connect/defresolver source-from-file [{:definition/keys [filename row col]}]
  {::pco/input [:definition/filename :definition/row (pco/? :definition/col)]
   ::pco/output [{:definition/source [:text/contents :text/range]}]}

  (p/let [source (helpers/read-file filename)
          top-blocks (helpers/top-blocks source)]
    (when-let [[range text] (helpers/top-block-for top-blocks [row (or col 0)])]
      {:definition/source {:text/contents text
                           :text/range range}})))

(def resolvers [join-paths-resolver seed-filename file-exists-resolver position-resolver
                existing-filename clojure-filename file-from-clr existing-row-col
                resolver-for-ns-only resolver-for-stacktrace
                source-from-contents source-from-file])
