(ns duck-repled.python.blocks
  (:require [duck-repled.tree-sitter :as ts]
            [com.wsscode.pathom3.connect.operation :as pco]
            [duck-repled.connect :as connect]
            [duck-repled.editor-helpers :as editor-helpers]
            [clojure.string :as str]
            [promesa.core :as p]
            ["path" :as path]))

(defn- to-range [^js node]
  (let [start (.-startPosition node)
        end (.-endPosition node)]
    [[(.-row start) (.-column start)]
     [(.-row end) (.-column end)]]))

(defn- get-node-block [^js node]
  (let [row (.. node -endPosition -row)]
    (loop [last-node node
           curr-node node]
      (if-let [node-row (some-> curr-node .-endPosition .-row)]
        (cond
          (some-> (.-parent curr-node) (= "module"))
          last-node

          (and (= node-row row) (-> curr-node .-type #{"expression_statement" "block" "import_statement" "import_from_statement"} not))
          (recur curr-node (.-parent curr-node))

          :else
          last-node)
        last-node))))

(pco/defresolver python-ts [{:keys [parsers]} {:text/keys [contents range language]}]
  {::pco/output [:python/node :python/captures]}
  (when (= :python language)
    (let [parsed (.parse ts/py-parser contents)
          [[r-start c-start] end] range
          [r-end c-end] end
          node (ts/root parsed)
          captures (ts/captures node
                                ts/any-query
                                {:row r-start :column (if (zero? c-start) 1 (dec c-start))}
                                {:row r-end :column c-end})]
      (swap! parsers conj parsed)
      {:python/node node
       :python/captures (mapv #(.-node %) captures)})))

(defn- find-block [captures]
  (if-let [import-node (->> captures
                            (filter #(#{"import_from_statement" "import_statement"} (.-type %)))
                            first)]
    {:text/contents (.-text import-node)
     :text/range (to-range import-node)}
    (let [last-node (last captures)
          parent-node (.-parent last-node)
          node-to-consider (cond
                             (-> parent-node .-type (= "call"))
                             parent-node

                             (-> last-node .-type (= "argument_list"))
                             parent-node

                             (-> parent-node .-type (= "argument_list"))
                             (.. last-node -parent -parent)

                             :else
                             last-node)
          block-node (get-node-block node-to-consider)]
      {:text/contents (.-text block-node)
       :text/range (to-range block-node)})))

(pco/defresolver block [{:keys [python/node python/captures text/range]}]
  {::pco/output [{:text/block [:text/contents :text/range]}]}

  {:text/block (find-block captures)})

(defn- find-root-node [^js node]
  (let [parent (-> node .-parent delay)]
    (cond
      (#{"function_definition" "class_definition"} (.-type node)) node
      @parent (recur @parent)
      :no-parent node)))

(defn- make-patch-code [^js method, ^js class]
  (let [method-name (-> method .-children second .-text)
        class-name (-> class .-children second .-text)]
    (str "\n" class-name "." method-name " = " method-name)))

(connect/defresolver top-block [{:keys [python/node text/range]}]
  {::pco/output [{:text/top-block [:text/contents :text/range :repl/watch-point]}]}
  (let [[[r-start c-start] end] range
        [r-end c-end] end
        captures (ts/captures node
                              ts/any-query
                              {:row r-start :column (if (zero? c-start) 1 c-start)}
                              {:row r-end :column (if (zero? c-end) 1 c-end)})
        top-block-node (-> captures
                           last
                           .-node
                           (find-root-node))
        class-definition (when (-> top-block-node .-type (= "function_definition"))
                           (find-root-node (.-parent top-block-node)))
        is-class-def? (some-> class-definition .-type (= "class_definition"))
        additional-content (when is-class-def?
                             (make-patch-code top-block-node class-definition))]
    (if (or is-class-def? (-> top-block-node .-type (= "class_definition")))
      {:text/top-block (cond-> {:text/contents (str (.-text top-block-node) additional-content)
                                :text/range (to-range top-block-node)
                                :repl/watch-point ""}
                         (and class-definition (not is-class-def?)) (dissoc :repl/watch-point))}
      {:text/top-block (find-block (map #(.-node %) captures))})))

(connect/defresolver selection [{:keys [text/contents text/range text/language]}]
  {::pco/output [{:text/selection [:text/contents :text/range]}]
   ::pco/priority 10}
  (when (= :python language)
    (when-let [text (editor-helpers/text-in-range contents range)]
      (let [lines (.split text #"\n")
            biggest (->> lines
                         (remove #(= "" %))
                         (map #(count (re-find #"^\s*" %)))
                         (apply min))]

        {:text/selection
         {:text/range range
          :text/contents (->> lines
                              (map #(str/replace % (re-pattern (str "^\\s{" biggest "}")) ""))
                              (str/join "\n"))}}))))

(def resolvers [python-ts block top-block selection])
