(ns orbit.nrepl.evaluator
  (:require [orbit.nrepl.bencode :as bencode]
            [orbit.evaluation :as eval]
            [clojure.edn :as edn]
            [promesa.core :as p]
            [orbit.serializer :as serializer]
            [orbit.meta-helper :as meta-helper]
            ["net" :as net]))

(defn- eval-op [command opts]
  (let [id (-> "eval" gensym str)
        op {:op "eval"
            :code (meta-helper/wrapped-command command (:col opts 0))
            ; :nrepl.middleware.print/stream? 1
            :id id}]
    (cond-> op
      (:namespace opts) (assoc :ns (:namespace opts))
      (:filename opts) (assoc :file (:filename opts))
      ;; We don't decrement row here because of wrapping
      (:row opts) (assoc :line (:row opts 0)))))

(defrecord REPL [^js conn state]
  eval/REPL
  (-evaluate [this command opts]
    (let [session-id (get-in @state [:sessions (get-in opts [:options :kind] :clj/eval)])
          op (if-let [nrepl-op (-> opts :options :op)]
               (assoc command :op nrepl-op :id (str (gensym "op")))
               (eval-op command opts))
          op (cond-> op session-id (update :session #(or % session-id)))
          promise (p/deferred)]
      (swap! state assoc-in [:pending (:id op)] {:promise promise
                                                 :opts opts})
      (.write conn (js/Buffer.from (bencode/encode op)) "binary")
      promise))

  (-break [_]
    (let [session-id (-> @state :sessions :clj/eval)]
      ; (prn :BREAK session-id)
      (.write conn (bencode/encode {:op :interrupt :session session-id}) "binary")))

  (-close [_] (.destroy conn)))

(defn- treat-socket-output! [{:keys [decode! buffer val treat on-output]}]
  (if (= :closed val)
    (on-output nil)
    (when val
      (swap! buffer subvec 1)
      (doseq [result (decode! val)]
        (treat result)))))

(def ^:private detection (str "#?("
                              ":bb :bb "
                              ":joker :joker "
                              ":clje :clje "
                              ":cljs :cljs "
                              ":cljr :cljr "
                              ":clj :clj "
                              ":default :unknown"
                              ")"))

(defn- connect-repl! [host port]
  (let [promise (p/deferred)]
    (let [buffer (atom [])
          ^hs conn (. net createConnection port host)]
      (.on conn "connect" #(p/resolve! promise {:buffer buffer :conn conn}))
      (.on conn "data" #(swap! buffer conj (str %)))
      (.on conn "error" #(p/reject! promise (. ^js % -errno)))
      (.on conn "close" #(reset! buffer [:closed]))
      promise)))

(defn- send-result! [id promise value]
  (let [decoded (-> value serializer/deserialize :result)]
    (p/resolve! promise (assoc decoded :id id))))

(defn- treat-output [state buffer contents]
  (when (seq contents) (reset! buffer []))
  (let [decode (:decoder @state)
        on-output (:on-output @state)]
    (if (= [:closed] contents)
      (do
        (remove-watch buffer :nrepl-evaluator)
        (on-output nil))
      (doseq [row contents
              decoded (decode row)
              :let [id (get decoded "id")
                    statuses (get decoded "status")
                    clearing-promise (fn [f]
                                       (f)
                                       (swap! state update :pending dissoc id))]]
        ; (prn :DEC decoded)
        (when-let [{:keys [promise] :as pending} (get-in @state [:pending id])]
          (cond
            (contains? decoded "out")
            (on-output {:out (get decoded "out")})

            (contains? decoded "err")
            (let [err (get decoded "err")]
              (swap! state update-in [:pending id :error] (fnil conj []) err)
              (on-output {:err err}))

            (contains? decoded "value")
            (clearing-promise #(send-result! id promise (get decoded "value")))

            (contains? decoded "ex")
            (clearing-promise
             #(p/resolve! promise
                          {:id id
                           :error (serializer/->RawData (str (get decoded "ex")
                                                             ": "
                                                             (:error pending)))}))

            (some #{"namespace-not-found"} statuses)
            (clearing-promise
             #(p/resolve!
               promise {:id id
                        :error (ex-info (str "Namespace " (-> pending :opts :namespace)
                                             " not found. Maybe you neeed to load-file,"
                                             " or evaluate the ns form?")
                                        {:not-found (-> pending :opts :namespace)})}))

            (some #{"interrupted"} statuses)
            (clearing-promise #(p/resolve! promise {:error (ex-info "Evaluation interrupted" {})
                                                    :id id}))

            (some #{"done"} statuses)
            (clearing-promise #(p/resolve! promise {:result decoded :id id}))))))))


(defn connect! [host port on-output]
  (p/let [{:keys [conn buffer]} (connect-repl! host port)
          state (atom {:on-output on-output
                       :decoder (bencode/decoder)})
          evaluator (->REPL conn state)
          _ (add-watch buffer :nrepl-evaluator #(treat-output state buffer %4))
          clone-op {:options {:op :clone} :plain true}
          sessions (p/all [(eval/evaluate evaluator {} clone-op)
                           (eval/evaluate evaluator {} clone-op)])
          #_#_
          repl-kind (-> (eval/evaluate evaluator detection)
                        (p/then :result)
                        (p/catch (constantly :unknown)))]
    (swap! state assoc :sessions {:clj/eval (get-in sessions [0 "new-session"])
                                  :clj/aux (get-in sessions [1 "new-session"])})
    evaluator))
