(ns orbit.meta-helper
  (:require [clojure.java.io :as io]
            [clojure.tools.reader :as r]
            [clojure.test :refer [testing]]
            [clojure.string :as str]
            [clojure.walk :as walk]))

(def wrap-js-patches
  `[(some-> object# .-then)
    (let [id# (str (gensym "promise-"))]
      (.then object# (fn [result#]
                       (tap> [{:orbit.patch/id id#}
                              (to-ebn {:orbit.patch/result result#})])))
      (.catch object# (fn [result#]
                        (tap> [{:orbit.patch/id id#}
                               (to-ebn {:orbit.patch/error result#})])))
      (to-meta# {:orbit.patch/id id#}
              (tagged-literal (-> object# type .-name (or "promise") symbol)
                              'pending)))

    (cljs.core/object? object#) (->> object#
                                     cljs.core/js-keys
                                     (map (fn [k#] [(keyword k#) (aget object# k#)]))
                                     (into {})
                                     to-ebn
                                     (str "#2ejs"))
    (cljs.core/array? object#) (->> object#
                                    (into [])
                                    to-ebn
                                    (str "#2ejs"))])

(def wrap-java-patches
  `(let [conv-no-coll# (fn [object#]
                         (if (class? object#)
                           (if (.isArray object#)
                             [(-> object# .getComponentType pr-str symbol
                                  (with-meta {:orbit.ui/lazy true}))]
                             (with-meta (symbol (.getName object#))
                               {:orbit.ui/lazy true}))
                           object#))
         convs# (fn [object#]
                  (if (coll? object#)
                    (walk/postwalk conv-no-coll# object#)
                    (conv-no-coll# object#)))]
     (cond
       (class? object#)
       (let [bean# (bean object#)]
         {:orbit.ui.reflect/info
          [{:title "Constructors"
            :contents
            (->> bean#
                 :declaredConstructors
                 (map (fn [m#]
                        (let [mod# (.getModifiers m#)]
                          {:public? (java.lang.reflect.Modifier/isPublic mod#)
                           :params (map convs# (.getParameterTypes m#))})))
                 (filter :public?)
                 (mapv (fn [m#]
                         (apply list (symbol (str (.getSimpleName object#) ".")) (:params m#)))))}
           {:title "Methods"
            :contents
            (->> bean#
                 :declaredMethods
                 (map (fn [m#]
                        (let [mod# (.getModifiers m#)]
                          {:public? (java.lang.reflect.Modifier/isPublic mod#)
                           :static? (java.lang.reflect.Modifier/isStatic mod#)
                           :params (map convs# (.getParameterTypes m#))
                           :ret (.getReturnType m#)
                           :name (.getName m#)})))
                 (filter :public?)
                 (sort-by :name)
                 (mapv (fn [m#]
                         (if (:static? m#)
                           (cons (symbol (str (.getSimpleName object#) "/" (:name m#))) (:params m#))
                           (->> (:params m#)
                                (cons 'this)
                                (cons (symbol (str "." (:name m#)))))))))}]})
       :object
       {:orbit.ui.reflect/info
        [{:title "Fields" :contents (->> object# bean (reduce merge {}) convs#)}]})))


(defn- apply-language-inspectors [form]
  (walk/postwalk-replace
   {`(comment "RawMeta") `(CONDITIONALSPLICE#
                            (:clj [~@wrap-java-patches]
                             :default [comment]))}
   form))

(defmacro contents-for-fn
  ([source-file] (slurp (io/resource source-file)))
  ([source-file fn-name]
   (let [re (re-pattern (str "(?s).*\\(defn\\s+(" fn-name "\\b)"))]
     (-> source-file
         io/resource
         slurp
         (str/replace re "(clojure.core/fn $1 ")
         (str/replace #"#\?@" "CONDITIONALSPLICE#")
         r/read-string
         pr-str
         (->> (str "`"))
         load-string
         apply-language-inspectors
         pr-str
         (str/replace #"CONDITIONALSPLICE[^\s]* " "#?@")
         (str/replace-first #"cond meta.*?\b"
                            (str "cond "
                                 "#?@(:cljs " wrap-js-patches ")"
                                 " meta__"))))))
