(ns orbit.nrepl.bencode
  (:require ["buffer" :refer [Buffer]]
            [clojure.string :as str]))

#?(:cljs
   (defn bytelength [string-ish]
     (.byteLength Buffer string-ish)))

(defn encode
  "Encodes a map/vector/string/number into BEncode format"
  [this]
  (cond
    (number? this) (str "i" this "e")
    (string? this) (str (bytelength this) ":" this)
    (keyword? this) (-> this str (str/replace-first #":" "") encode)
    (symbol? this) (-> this str encode)
    (map? this) (str "d"
                     (->> this
                          (map (fn [[k v]] (str (encode k) (encode v))))
                          (str/join ""))
                     "e")
    (coll? this) (str "l" (->> this (map encode) (str/join "")) "e")
    :else (throw (ex-info "Can't encode this object" {:object this
                                                      :type (type this)}))))

#_
(defn- decode-fragment [fragment acc]
  (let [f (-> fragment first str)]
    (cond
      (= "i" f)
      (if-let [[res value] (re-find #"i(\-?\d+)e" fragment)]
        (recur (subs fragment (count res)) (conj acc (js/parseInt value)))
        [fragment acc])

      (= "l" f)
      (let [[rest inner] (decode-fragment (subs fragment 1) [])]
        (if (= "e" (first rest))
          (recur (subs rest 1) (conj acc inner))
          [fragment acc]))

      (= "d" f)
      (let [[rest inner] (decode-fragment (subs fragment 1) [])]
        (if (= "e" (first rest))
          (recur (subs rest 1) (conj acc (apply hash-map inner)))
          [fragment acc]))

      (re-find #"\d" f)
      (let [[_ c] (re-find #"^(\d+):" fragment)
            chars (js/parseInt c)
            start (-> c count inc)
            value (str (byteslice fragment start (+ start chars)))]
        (if (->> value bytelength (= chars))
          (recur (subs fragment (-> value count (+ start))) (conj acc value))
          [fragment acc]))

      :else
      [fragment acc])))

#_
(defn old-decoder
  []
  (let [state (atom "")]
    (fn [fragment]
      (swap! state str fragment)
      (let [[rest parsed] (decode-fragment @state [])]
        (reset! state rest)
        parsed))))

(declare decode*)
(defn- parse-str [num-bytes acc fragment]
  (let [frag-size (.-length fragment)]
    (if (>= frag-size num-bytes)
      (let [current (.slice fragment 0 num-bytes)
            next-frag (.slice fragment num-bytes frag-size)]
        (merge-with concat
                    {:results [(.toString (.concat Buffer #js [acc current]) "utf-8")]}
                    (decode* next-frag)))
      (let [acc (.concat Buffer #js [acc fragment])]
        {:next (fn decode-partial-str2 [next-fragment] (parse-str (- num-bytes frag-size)
                                                                  acc
                                                                  next-fragment))}))))

(defn- ^:inline b-find [^js buffer char]
  (let [found (.indexOf buffer char)]
    (when (> found -1) found)))

(def ^:private empty-buffer (Buffer. 0 0))

(defn- parse-list [^js fragment acc next-parse]
  (let [result (next-parse fragment)
        res (:results result)
        new-res (delay (reduce conj! acc res))]
    (cond
      (::done result)
      (merge-with concat
                  {:results [(persistent! @new-res)]}
                  ((:next result) empty-buffer))

      res
      (let [fun (:next result)
            new-acc @new-res
            next (fn [next-fragment] (parse-list next-fragment new-acc fun))]
        {:next next})

      :no-new-result
      (let [fun (:next result)]
        {:next (fn [next-fragment] (parse-list next-fragment acc fun))}))))

(defn- parse-map [^js fragment acc next-key next-parse]
  (let [result (next-parse fragment)
        res (cond->> (:results result)
              (some? next-key) (cons next-key))
        [completes incompletes] (->> res
                                     (partition-all 2)
                                     (split-with #(-> % count (= 2))))
        incomplete (-> incompletes ffirst delay)
        new-res (->> completes
                     (reduce (fn [acc [k v]]
                               (assoc! acc k v))
                             acc)
                     delay)]
    (cond
      (::done result)
      (merge-with concat
                  {:results [(persistent! @new-res)]}
                  ((:next result) empty-buffer))

      res
      (let [fun (:next result)
            new-acc @new-res
            incomplete @incomplete
            next (fn [next-fragment] (parse-map next-fragment new-acc incomplete fun))]
        {:next next})

      :no-new-result
      (let [fun (:next result)
            incomplete @incomplete]
        {:next (fn [next-fragment] (parse-map next-fragment acc incomplete fun))}))))

(defn- parse-int [^js fragment acc]
  (if-let [idx (b-find fragment "e")]
    (merge-with concat
                {:results [(js/parseInt (.concat Buffer #js [acc (.slice fragment 0 idx)]))]}
                (decode* (.slice fragment (inc idx))))
    (let [sofar (.concat Buffer #js [acc fragment])]
      {:next (fn decode-partial-int [next-frag] (parse-int next-frag sofar))})))

(defn- decode* [^js fragment]
  (case (.toString fragment "utf-8" 0 1)
    "" {:next decode*}
    "i" (parse-int (.slice fragment 1) empty-buffer)
    "l" (parse-list (.slice fragment 1) (transient []) decode*)
    "d" (parse-map (.slice fragment 1) (transient {}) nil decode*)
    "e" {:results []
         :next (fn inner-decode* [new-fragment]
                 (decode* (.concat Buffer #js [(.slice fragment 1) new-fragment])))
         ::done ::done}

    ("0" "1" "2" "3" "4" "5" "6" "7" "8" "9")
    (if-let [idx (b-find fragment ":")]
      (parse-str (js/parseInt fragment)
                 empty-buffer
                 (.slice fragment (inc idx)))
      {:next (fn decode-partial-str1 [next-frag] (decode* (.concat Buffer #js [fragment next-frag])))})

    (throw (ex-info "Garbage while parsing bencode" {:fragment fragment}))))

(defn decoder
  "Starts a stateful decoder. It will return a function that accepts one parameter
(a string) and it'll try to decode it as a BEncode value. It'll return the BEncode
structures it finds, or an empty vector if it didn't found anything.

Ex:
(let [decode! (decoder)]
  (is (= [10] (decode! \"i10e\")))
  (is (= [] (decode! \"i1\")))
  (is (= [10] (decode! \"0e\"))))"
  []
  (fn [fragment]
    (decode* fragment)))

;; Performance tests
#_
(->> 100000 range pr-str encode (def encoded))
(defn hey-ho! []
  #_
  (time
   (let [decode! (decoder)]
     (->> encoded
          (partition-all 20 20)
          (map #(apply str %))
          (map decode!)
          last
          last
          last))))

#_
(let [decode! (decoder)
      chunks (->> encoded
                  (partition-all 20 20)
                  (map #(.from Buffer (apply str %))))]
  (time
   (do
     ; (js-debugger)
     (->
      (reduce (fn [{:keys [next]} chunk]
                (next chunk))
              {:next decode!}
              chunks))
      ; :results
      ; first
      ; count)
     nil)))
#_
(time
 (let [decode! (old-decoder)]
   (->> encoded
        (partition-all 20 20)
        (map #(apply str %))
        (map decode!)
        last
        last
        last)))
