(ns spock.swi
  (:require [instaparse.core :as insta]
            [spock.commons :as commons]
            [clojure.string :as str]))

(def parse-prolog
  (insta/parser
    "
<terms> = term (<', '> term)*
<term> = equality | atom | var | number | boolean | structure | list | string
equality = var <' = '>  term
list = <'[]'> | <'['> term (<', '> term)*  <']'>
structure = atom <'('> terms <')'>
string = <'\"'> (#\"[^\\\"]+\" | '\\\"')* <'\"'>
atom = atom-prefix letter* | \"'\" atom-sequence* \"'\"
<atom-sequence> = #\"(\\\\'|[^'])\"
var = var-prefix letter*
<atom-prefix> = #\"[a-z]\"
<letter> = #\"[\\d\\w_]\"
<var-prefix> = #\"[A-Z_]\"
number = #\"\\d+(\\.\\d+)?\"
boolean = 'true' | 'false'
"))

(defmulti parse-out first)
(defmethod parse-out :equality [[_ var thing]] [(parse-out var) (parse-out thing)])
(defmethod parse-out :list [[_ & params]] (mapv parse-out params))
(defmethod parse-out :structure [[_ & structs]] (map parse-out structs))
(defmethod parse-out :string [[_ & str]] (str/join "" str))
(defmethod parse-out :atom [[_ & var]]
  (let [var-name (str/join "" var)]
    (-> var-name
        str/lower-case
        (str/replace-first #"^'" "")
        (str/replace-first #"'$" "")
        (str/replace #"_" "-")
        symbol)))

(defmethod parse-out :var [[_ & var]]
  (let [var-name (str/join "" var)]
    (-> var-name
        str/lower-case
        (str/replace-first #"^_" "")
        (str/replace #"_" "-")
        keyword)))

(defmethod parse-out :number [[_ num]] (js/parseInt num))
(defmethod parse-out :boolean [[_ b]] (= b "true"))

(defonce streams (atom {:pos 0
                        :code "A = 10."
                        :out ""
                        :parsed []}))

(defn- normalize-stdout [out]
  (-> out
      (str/replace #"(\n|\s)+" " ")
      str/trim
      (str/replace #"\.$" "")))

(defn stdin-code [streams]
  (let [{:keys [pos code out next?]} @streams]
    (if (< pos (count code))
      (let [code (.charCodeAt code pos)]
        (swap! streams update :pos inc)
        code)
      (when next?
        (swap! streams
               (fn [streams]
                 (-> streams
                     (assoc :code "n"
                            :pos 0
                            :out ""
                            :next? false)
                     (update :parsed conj (normalize-stdout (:out streams))))))
        (stdin-code streams)))))

(defn stdout-code [streams char]
  (let [char (js/String.fromCharCode char)]
    (swap! streams #(-> %
                        (assoc :next? (and (not= char ".")
                                           (not= char "\n")))
                        (update :out str char)))))

(defn prepare-run [streams ^js module]
  (.. module -FS (init (partial stdin-code streams)
                       (partial stdout-code streams))))

(defn create-runtime!
  ([constructor] (create-runtime! constructor {}))
  ([constructor opts]
   (let [streams (atom {:pos 0
                        :code "A = 10."
                        :out ""
                        :parsed []})]
     (. (constructor (-> {:arguments ["swipl" "-q" "-x" "/src/wasm-preload/boot.prc" "--nosignals"]}
                         (merge opts)
                         (assoc :preRun #js [(partial prepare-run streams)]
                                :printErr #(swap! streams update :error str % "\n"))
                         clj->js))
       (then (fn [^js r]
               (.. r -FS (mkdirTree "wasm-preload/library"))
               #js {:runtime r
                    :streams streams}))))))

(defn use-library [^js runtime lib-name contents]
  (.. runtime -runtime -FS (writeFile (str "wasm-preload/library/" lib-name ".pl")
                                      contents))
  (.. runtime -runtime -prolog (call_string (str "use_module(library(" lib-name "))"))))

(defn query! [^js swi code]
  (let [streams (.-streams swi)]
    (reset! streams {:code code :pos 0 :out "" :parsed []})
    (.. swi -runtime -prolog (call_string "break"))
    (if-let [error (:error @streams)]
      (throw (ex-info "Query failed!" {:error error}))
      (->> (-> @streams :out normalize-stdout)
           (conj (:parsed @streams))
           (filter identity)))))

(defprotocol AsProlog (to-prolog [this]))

(defn- as-atom [unparsed]
  (str "'" (str/replace (str unparsed) #"\\" "\\\\") "'"))

(defn- as-struct [unparsed]
  (let [[head & tail] (commons/normalize-struct unparsed)]
    (str (as-atom head)
         "("
         (->> tail
              (map to-prolog)
              (str/join ", "))
         ")")))

(extend-protocol AsProlog
  string
  (to-prolog [this] (pr-str this))

  boolean
  (to-prolog [this] (pr-str this))

  number
  (to-prolog [this] (str this))

  Keyword
  (to-prolog [this] (-> this name (str/replace-first #"." (fn [s] (str/upper-case s)))))

  Symbol
  (to-prolog [this] (as-atom this))

  object
  (to-prolog [this]
    (cond
      (vector? this) (let [[bef aft] (split-with #(not= % '&) this)]
                       (str "["
                            (if (-> aft count (= 2))
                              (str (->> bef (map to-prolog) (str/join ", "))
                                   " | " (->> aft last to-prolog))
                              (->> this (map to-prolog) (str/join ", ")))
                            "]"))

      (let [mapped (map to-prolog this)])
      (str "["
           (->> this
                (map to-prolog)
                (str/join ", "))
           "]")
      (list? this) (as-struct this)
      :else (throw (ex-info "Unsupported Object" {:type (type this)})))))

(defrecord SWI [^js runtime]
  commons/SWIWrapper
  (call! [_ query] (.. runtime -runtime -prolog (call_string (to-prolog query)))))

(defn with-rules [runtime rules]
  (commons/with-rules (->SWI runtime) rules))

(defn assert-rules [runtime rules]
  (commons/assert-rules (->SWI runtime) rules))

(defn- get-val-for-key [key vars]
  (let [val (get vars key ::missing)]
    (cond
      (= val ::missing) [::missing key]
      (keyword? val) (get-val-for-key val vars)
      :else [val])))

(defn- unify-vars [vars]
  (if (some keyword? (vals vars))
    (->> vars
         (mapcat (fn [[k v]]
                   (let [[new-val possible-k] (get-val-for-key k vars)]
                     (if (= ::missing new-val)
                       [[k :_] [possible-k :_]]
                       [[k new-val]]))))
         (into {}))
    vars))

(defn- from-prolog [result-strings]
  (try
    (->> result-strings
         (map (fn [one-result]
                (try
                  (->> one-result
                       parse-prolog
                       (reduce #(let [parsed (parse-out %2)]
                                  (cond
                                    (vector? parsed) (conj %1 parsed)
                                    parsed (conj %1 {})))
                               {})
                       unify-vars)
                  (catch :default e
                    (println "Error parsing:" e)
                    :invalid-output))))
         (filter identity))
    (catch :default e
      (println "Error parsing:" e)
      :invalid-output)))

(defn solve [{:keys [runtime] :as opts} query]
  (let [[query from-prolog] (commons/prepare-solve from-prolog opts query)
        as-str (-> query to-prolog (str "."))]
    (->> as-str
         (query! runtime)
         from-prolog)))
