(ns tango.ui.elements
  (:require [clojure.string :as str]
            ["react" :as react]))

(defn Children [& contents]
  (into [:div.children] contents))

(defn Rows [& contents]
  (into [:div.rows] contents))

(defn Cols [& contents]
  (into [:div.cols] contents))

(defn Title [& contents]
  (into [:div.title] contents))

(defn WithClass [class child]
  [:div {:class class} child])

(defn open-close
  "Creates a link with open, or close, elements. If it's open, it'll show
children too. If closed, it won't show anything.

`:state` which contains `:open?` - it overrides the `:originally-open?` parameter.
`:originally-open?` says if it the chevron is supposed to be open the first
  time, if there's no `:open?` key on `state`

Returns `:open?`, if was supposed to be open, and link, which is the element"
  [state originally-open?]
  (let [open? (:open? @state originally-open?)
        link [:a.chevron {:class (if open? "opened" "closed")
                          :href "#"
                          :on-click (fn [e]
                                      (.preventDefault e)
                                      (.stopPropagation e)
                                      (swap! state assoc :open? (not open?)))}
              ""]]
    {:open? open? :link link}))

(def ^{:doc "A simple whitespace. Use with `[ui/Cols \"A\" ui/Whitespace \"B\"]`"}
  Whitespace (react/createElement "div" #js {:className "inline"} " "))

;; FIXME: don't use Atom API here
(defn Markdown
  "Reagent  element to render a Markdown text (defined by `string`). Can be used in interactive
results. Code blocks without any language will be assumed to be Clojure.

Usage - evaluate the following code:
```
^:tango/interactive {:html '[ui/Markdown \"_Hello_, **world!**\"]}
```"
  [string]
  [:div.markdown {:ref (fn [^js ref]
                         (when ref
                           (let [md (.. js/atom -ui -markdown)
                                 frag (.render md
                                               string
                                               #js {:renderMode "fragment"
                                                    :breaks false
                                                    :useDefaultEmoji true})
                                 dom (.convertToDOM md frag)]
                             (.applySyntaxHighlighting md dom
                                                       #js {:renderMode "fragment"
                                                            :syntaxScopeNameFunc (fn [lang]
                                                                                   (if (seq lang)
                                                                                     (str "source." (str/lower-case lang))
                                                                                     "source.clojure"))})
                             (.appendChild ref dom))))}])

Markdown
