(ns tango.ui.edn
  (:require [reagent.core :as r]
            ["react" :as react]
            [orbit.serializer :as serializer]
            [promesa.core :as p]
            [reagent.ratom :as ratom]
            [clojure.string :as str]
            [cljs.js :as j]
            [cljs.repl :as repl]
            [tango.ui.interactive :as int]
            [tango.integration.repl-helpers :as helpers]
            [tango.ui.elements :as ui]))

(declare as-html)

(defn- create-child-state [state & keys]
  (let [key-path (cons :child-state keys)]
    (-> (:state state)
        (swap! update-in key-path
               #(or % (r/atom (:child-state state {}))))
        (get-in key-path))))

(defn custom-cursor [ratom index]
  (let [result (:result @ratom)
        metadata (meta result)]
    (cond
      (or (:orbit.ui.reflect/info metadata) (:orbit.ui/lazy metadata))
      (ratom/make-reaction
       (fn []
         (if (= :meta index)
           (update @ratom :result meta)
           (update @ratom :result vary-meta dissoc
                   :orbit.ui.reflect/info :orbit.ui/lazy)))

       :on-set (fn [_ {:keys [result]}]
                 (swap! ratom update :result
                        #(if (= :meta index)
                           (with-meta % result)
                           (with-meta result (meta %))))))

      (serializer/tagged-literal-with-meta? result)
      (ratom/make-reaction
       (fn []
         (if (= index :tag)
           (update @ratom :result #(.-tag %))
           (update @ratom :result #(.-form %))))

       :on-set (fn [_ {:keys [result]}]
                 (swap! ratom update :result
                        #(if (= :tag index)
                           (serializer/tagged-literal-with-meta result (.-form %))
                           (serializer/tagged-literal-with-meta (.-tag %) result)))))

      (map? result)
      (let [[index what] index
            idx-map (-> result seq (nth index) first atom)]
        (ratom/make-reaction
         #(if (= what :key)
            (assoc @ratom :result @idx-map)
            (update @ratom :result get @idx-map))
         :on-set (fn [_ {:keys [result]}]
                   (if (= :key what)
                     (let [old-key @idx-map]
                       (reset! idx-map result)
                       (swap! ratom update :result
                              #(-> %
                                   (assoc result (get % old-key))
                                   (dissoc old-key))))
                     (swap! ratom assoc-in [:result @idx-map]
                            result)))))

      (set? result)
      (let [idx-map (atom (nth (seq result) index))]
        (ratom/make-reaction
         #(update @ratom :result get @idx-map)
         :on-set (fn [_ {:keys [result]}]
                   (let [old-val @idx-map]
                     (reset! idx-map result)
                     (swap! ratom update :result
                            #(-> %
                                 (disj old-val)
                                 (conj result)))))))

      (list? result)
      (ratom/make-reaction
       #(update @ratom :result nth index)
       :on-set (fn [_ {:keys [result]}]
                 (swap! ratom update :result
                        (fn [lst]
                          (let [[s e] (split-at index lst)]
                            (apply list (concat s (->> e rest (cons result)))))))))
      :else (ratom/make-reaction
             #(update @ratom :result get index)
             :on-set (fn [_ {:keys [result]}]
                       (swap! ratom assoc-in [:result index] result))))))

(defn- update-state [state index-or-atom merge-data child-state]
  (let [new-eval-result (if (satisfies? IDeref index-or-atom)
                          index-or-atom
                          (custom-cursor (:eval-result state) index-or-atom))]
    (-> state
        (merge merge-data)
        (assoc :state child-state)
        (assoc :eval-result new-eval-result))))

(defn- clj-stack-row [[class method file row] state]
  (let [eval-result (:eval-result state)
        clj-file? (re-find #"\.(clj.?|bb)$" (str file))
        file (if (re-find #"(NO.FILE|form\-init\d+)" file)
               (:filename @eval-result "NO-FILE")
               file)]
    [ui/Cols  "  at "
     (if clj-file?
       [ui/WithClass ["stack" "clojure"]
        [:<>
         (str (demunge (str class)) " (")
         [ui/Link (fn [evt]) []
          (str file ":" row)]
         ")"]]
       [ui/WithClass ["stack" "non-clojure"]
        [ui/Cols
         (str class)
         "."
         (str method)
         (str " (" file ":" row ")")]])]))

(defn- demunge-js-name [js-name]
  (if (str/starts-with? js-name "global.")
    js-name
    (-> js-name
        (str/replace #"\$" ".")
        (str/replace #"(.*)\.(.*)$" "$1/$2")
        demunge)))

(defn goto-stacktrace [e state file row col]
  (.preventDefault e)
  (.stopPropagation e)
  (p/let [editor-state @(:editor-state state)
          eql (-> editor-state :editor/features :eql)
          open (-> editor-state :editor/callbacks :open-editor)
          result (eql {:ex/filename file
                       :ex/row (js/parseInt row)
                       :ex/col (some-> col js/parseInt)}
                      [:definition/filename :definition/row :definition/col])]
    (open {:file-name (:definition/filename result)
           :line (:definition/row result)
           :column (:definition/col result)})))

(defn- generic-stacktrace-row [{:keys [what file row col]} state]
  [ui/Cols
   "  at "
   [ui/WithClass ["stack"]
    [:<>
     (when what (str what " ("))
     [ui/Link #(goto-stacktrace % state file row col) []
      (str file ":" row (when col (str ":" col)))]
     (when what ")")]]])

(defn- generic-stacktrace [state]
  (into [ui/Rows]
        (map #(generic-stacktrace-row % state)
             @(:result state))))

(defn- cljs-stack-row [stack-row state]
  (when (and (re-find #"^\s+at" stack-row)
             (not (re-find #"\<eval\>" stack-row)))
    (let [found (re-find #"at (.*) \((.*?):(\d+):?(\d+)?\)" stack-row)
          [_ what file row col] (if found
                                  found
                                  (concat [nil nil]
                                          (rest (re-find #"at (.*?):(\d+):?(\d+)?" stack-row))))]
      [ui/Cols
       "  at "
       [ui/WithClass ["stack" "js"]
        [:<>
         (if what (demunge-js-name what) "<unknown>")
         " ("
         [ui/Link #(goto-stacktrace % state file row col) []
          (str file ":" row (when col (str ":" col)))]
         ")"]]])))

(defn- error-result [state]
  (let [result (:result state)
        root? (:root? state)
        {:keys [open? link]} (when root? (ui/open-close (:state state) false))
        tag-part (-> @result .-tag pr-str)
        form-part (.-form @result)
        form-part (cond-> form-part
                    (serializer/tagged-literal-with-meta? form-part) .-form)
        eval-result (:eval-result state)
        child-result (dissoc form-part :via :trace :cause)
        form-ratom (r/atom (assoc @eval-result :result child-result))
        child-state (update-state state form-ratom {:root? root?}
                                  (create-child-state (assoc state :child-state {:open? false})))
        stack-row (if (-> form-part :trace string?)
                    cljs-stack-row
                    clj-stack-row)]
    [ui/Rows
     [ui/WithClass (when root? "title") (str "#error " (or (:cause form-part)
                                                           (-> form-part :via first :message)
                                                           (:message form-part)))]
     (when root?
       [:<>
        (when (seq child-result)
          [ui/Children [as-html child-state]])

        (when-let [via (:via form-part)]
          (->> via
               (map (fn [data]
                      (let [data (if (serializer/tagged-literal-with-meta? data)
                                   (.-form data)
                                   data)]
                        [ui/Cols (when-let [type (-> data :type)]
                                   (str (pr-str type) ": "))
                              (-> data :message str)])))
               (into [ui/Children ui/Space "Via:"])))

        (when-let [stack (:trace form-part)]
          (->> (cond-> stack (string? stack) str/split-lines)
               (map (fn [row] [stack-row row state]))
               (into [ui/Children ui/Space "Stacktrace:"])))])]))

(defn- tagged [state]
  (if (-> @(:result state) .-tag (= 'error))
    (error-result state)
    (let [result (:result state)
          root? (:root? state)
          child-state (create-child-state (assoc state :child-state {:open? true}))
          {:keys [open? link]} (when root? (ui/open-close (:state state) false))
          tag-part (-> @result .-tag pr-str)
          form-cursor (custom-cursor (:eval-result state) :form)
          parent-state (update-state state form-cursor {:root? false} child-state)]

      (cond
        (not root?) [ui/Rows [ui/Cols "#" tag-part " " [as-html parent-state]]]
        open? [ui/Rows [ui/Cols link "#" tag-part]
               [ui/Children [as-html (update-state state
                                                   form-cursor
                                                   {:root? true}
                                                   child-state)]]]
        :closed? [ui/Cols link "#" tag-part " " [as-html parent-state]]))))

(defn leaf [state]
  [ui/Rows (pr-str @(:result state))])

(defn raw-data [state]
  [ui/Rows (-> state :result deref :data)])

(defn- collection [open state close]
  (let [children-states (->> state
                             :result
                             deref
                             (map-indexed (fn [i _]
                                            [(custom-cursor (:eval-result state) i)
                                             (create-child-state state i)])))
        children (->> children-states
                      (map (fn [[cursor child-state]]
                             [as-html
                              (update-state state
                                            cursor
                                            {:root? true}
                                            child-state)])))
        parent (->> children-states
                    (map (fn [[cursor child-state]]
                           [as-html (update-state state
                                                  cursor
                                                  {:root? false}
                                                  child-state)]))
                    (interpose ui/Whitespace)
                    (into [:<>]))
        {:keys [open? link]} (ui/open-close (:state state) false)]

    (cond
      (-> state :root? not) [ui/Cols open parent close]
      open? [ui/Rows [ui/Cols link " " open parent close]
             (into [ui/Children] children)]
      :closed? [ui/Rows [ui/Cols link " " open parent close]])))

(defn as-vector [state]
  (collection "[" state "]"))

(defn as-set [state]
  (collection "#{" state "}"))

(defn as-list [state]
  (collection "(" state ")"))

(def Comma (react/createElement "div" #js {:className "inline"} ", "))
(defn as-map [state]
  (let [ratom (:eval-result state)
        children-states (->> state
                             :result
                             deref
                             (map-indexed (fn [i _]
                                            [(custom-cursor ratom [i :key])
                                             (custom-cursor ratom [i :val])
                                             (create-child-state state i :key)
                                             (create-child-state state i :val)])))
        children (->> children-states
                      (map (fn [[k-cursor v-cursor k-state v-state]]
                             [:<>
                              (ui/WithClass
                               ["map-key" "opened"]
                               [as-html
                                (update-state state
                                              k-cursor
                                              {:root? true}
                                              k-state)])
                              [as-html
                               (update-state state
                                             v-cursor
                                             {:root? true}
                                             v-state)]]))
                      (interpose ui/Whitespace))
        parent (->> children-states
                    (map (fn [[k-cursor v-cursor k-state v-state]]
                           [:<>
                            (ui/WithClass
                             ["map-key" "closed"]
                             [as-html (update-state state
                                                    k-cursor
                                                    {:root? false}
                                                    k-state)])
                            ui/Whitespace
                            [as-html (update-state state
                                                   v-cursor
                                                   {:root? false}
                                                   v-state)]]))
                    (interpose Comma)
                    (into [:<>]))
        {:keys [open? link]} (ui/open-close (:state state) false)]

    (cond
      (-> state :root? not) [ui/Cols "{" parent "}"]
      open? [ui/Rows [ui/Cols link "{" parent "}"]
             (into [ui/Children] children)]
      :closed? [ui/Rows [ui/Cols link "{" parent "}"]])))

(defn- find-right-seq [state key]
  (->> @state
       :result
       keys
       (map-indexed (fn [i k] [i (= k key)]))
       (filter peek)
       ffirst))

(defn- reflection-info-details [{:keys [eval-result] :as state}]
  (let [index (find-right-seq eval-result :orbit.ui.reflect/info)
        cursor (custom-cursor eval-result [index :val])
        new-state (update-state state cursor {} (:state state))]
    (->> @cursor
         :result
         (map-indexed (fn [parent-idx {:keys [title contents]}]
                        (let [curr-cursor (custom-cursor (:eval-result new-state)
                                                         parent-idx)
                              funs-cursor (custom-cursor curr-cursor
                                                         [(find-right-seq curr-cursor :contents)
                                                          :val])]
                          (when-not (map? contents)
                            (->> contents
                                 (map-indexed (fn [idx _]
                                                [as-html (update-state
                                                          new-state
                                                          (custom-cursor funs-cursor idx)
                                                          {:root? true}
                                                          (create-child-state new-state
                                                                              parent-idx
                                                                              idx))]))
                                 (into [ui/Rows ui/Space [ui/Title title]]))))))
         (into [:<>]))))

(defn MoreInfo [callback]
  [ui/Link callback ["more-info"]
    ""])

(defn- reflect [{:keys [eval-result] :as state}]
  (let [reflection-info (custom-cursor eval-result :meta)
        without-meta (custom-cursor eval-result :data)
        ratom (:state state)
        reflection-info-open? (:reflection-info-open? @ratom)]

    [ui/Cols [as-html (update-state state
                                    without-meta
                                    {:root? (and (:root? state)
                                                 (not reflection-info-open?))}
                                    (:state state))]
     (when (:root? state)
       [:<>
        ui/Whitespace
        (when (and (:root? state) (not reflection-info-open?) (not (:open? @ratom)))
          [MoreInfo #(swap! ratom assoc :reflection-info-open? true)])
        (when (and (:root? state) reflection-info-open? (not (:open? @ratom)))
          [ui/Children
           [ui/Cols
            [ui/Title "More Info"]
            ui/Whitespace
            [ui/Link #(swap! ratom assoc :reflection-info-open? false) []
             "Hide"]]
           [reflection-info-details (update-state state
                                                  reflection-info
                                                  {}
                                                  (:state state))]])])]))

(defn- lazy [{:keys [eval-result] :as state}]
  (let [reflection-info (custom-cursor eval-result :meta)
        metadata (custom-cursor eval-result :meta)
        without-meta (custom-cursor eval-result :data)
        ratom (:state state)
        reflection-info-open? (:reflection-info-open? @ratom)]
    [ui/Cols
     [as-html (update-state state
                            without-meta
                            {:root? (not reflection-info-open?)}
                            (:state state))]
     (when (:root? state)
       [:<>
        ui/Whitespace
        [MoreInfo (fn []
                    (p/let [eql (-> state :editor-state deref :editor/features :eql)
                            result (eql {:v {:text/contents (-> @eval-result :result pr-str)}}
                                        [{:v [:repl/result]}])
                            new-meta (-> result :v :repl/result :result meta)]
                      (swap! ratom assoc :reflection-info-open? true)
                      (swap! metadata assoc :result new-meta)))]])]))

(defn- wrapped-error [state]
  (let [child (custom-cursor (:eval-result state) 0)
        new-state (update-state state child {} (:state state))]
    [ui/WithClass ["error"]
     [as-html new-state]]))

(defn- text-with-stacktrace [state texts-or-traces]
  (into [ui/Rows]
        (for [row texts-or-traces]
          (if (map? row)
            [ui/Children
             [ui/Cols
              "at "
              [ui/Link #(goto-stacktrace % state (:file row) (:line row) (:column row))
               []
               (str (:resource-name row (:file row))
                    ":" (:line row)
                    (when-let [col (:column row)]
                      (str ":" col)))]]]
            row))))

#_(nil 10)
(defn- shadow-error [error state]
  (let [traces (re-seq #" File: (.*?):(\d+):(\d+)" error)]
    (->> traces
         (map (fn [[_ file row col?]]
                (cond-> {:file file :line row}
                  col? (assoc :column col?))))
         (into [[ui/Title "Errors found in compilation process"]
                ui/Space
                [ui/Cols error]])
         (text-with-stacktrace state))))

(defn- shadow-warnings [warnings state]
  (let [all-warnings (for [warning warnings]
                       [[ui/WithClass "error" (:msg warning)]
                        (when-let [e (:source-excerpt warning)]
                          [ui/Code
                           (->> e
                                :before
                                (map #(vector :div.block (if (seq %) % " ")))
                                (into [:<>]))
                           [ui/Cols (:line e)]
                           [ui/Cols (str (->> " "
                                              (repeat (-> warning :column dec))
                                              (apply str))
                                         "^")]])
                        warning
                        ui/Space])]
    (->> all-warnings
         (mapcat identity)
         (into [[ui/Title "Warnings"]
                ui/Space])
         (text-with-stacktrace state))))

(defn- shadow-errors [state]
  (let [eval-result @(:eval-result state)
        result (:result eval-result)
        warnings (:orbit.shadow/warnings result)
        norm-warnings (map (fn [warning]
                             (if (:file warning)
                               warning
                               (-> warning
                                   (assoc :file (:editor/filename eval-result))
                                   (update :line + -1 (-> eval-result :editor/data :range ffirst))
                                   (update :column + (-> eval-result :editor/data :range first second)))))
                           warnings)]
    (cond
      warnings (shadow-warnings norm-warnings state)

      (-> result :orbit.shadow/error (= :no-runtime-selected))
      [ui/Cols "No javascript runtime running right now"]
      :else (shadow-error (:orbit.shadow/errors result) state))))

(defn- as-html [state]
  (let [result (-> state :eval-result (r/cursor [:result]))
        new-state (assoc state :result result)
        metadata (meta @result)]
    (when-let [id (:orbit.patch/id metadata)]
      (swap! (:patches state) assoc id result))
    (cond
      (:orbit.ui.reflect/info metadata) (reflect new-state)
      (:orbit.ui/lazy metadata) (lazy new-state)
      (:orbit.shadow/error metadata) (shadow-errors new-state)
      (:tango/interactive metadata) (int/interactive new-state)
      (:tango/wrapped-error metadata) (wrapped-error new-state)
      (:tango/generic-stacktrace metadata) (generic-stacktrace new-state)
      (instance? serializer/RawData @result) (raw-data new-state)
      (serializer/tagged-literal-with-meta? @result) (tagged new-state)
      (map? @result) (as-map new-state)
      (vector? @result) (as-vector new-state)
      (set? @result) (as-set new-state)
      (coll? @result) (as-list new-state)
      :else (leaf new-state))))

(defn for-result [result editor-state]
  (r/with-let [local-state (r/atom {})
               patches (atom {})
               result (if (:error result)
                        (assoc result :result ^:tango/wrapped-error [(:error result)])
                        result)
               result-a (r/atom result)
               new-eql (helpers/prepare-new-eql editor-state)
               new-editor-state (atom (update @editor-state :editor/features assoc
                                             :eql new-eql
                                             :original-eql (-> @editor-state
                                                               :editor/features
                                                               :eql)))]
    ^{:patches patches}
    [as-html {:eval-result result-a
              :editor-state new-editor-state
              :patches patches
              :state local-state
              :root? true}]))
