(ns tango.state)

(def ^:private empty-state {:states (sorted-map) :commands {}})

(defonce ^:private connections (atom empty-state))

(defn get-state [lang]
  (let [config-state (get-in @connections [:states :Config])
        config? (some-> config-state deref :editor/features :is-config?)]
    (if (and config-state (config?))
      config-state
      (get-in @connections [:states lang]))))

(defn add [lang state]
  (swap! connections assoc-in [:states lang] state)
  state)

(defn get-a-state []
  (-> @connections :states vals last))

(defn set-commands [lang commands]
  (let [old-commands (-> @connections :commands keys)]
    (doseq [command old-commands]
      (swap! connections update :commands
             #(let [without (update % command dissoc lang)]
                (if (-> without (get command) seq)
                  without
                  (dissoc % command)))))

    (doseq [[cmd {:keys [command]}] commands]
      (swap! connections assoc-in [:commands cmd lang] command))))

(defn all-commands []
  (for [[cmd-key languages] (:commands @connections)
        :let [callbacks (:editor/callbacks @(get-state :Config))
              editor-data (:editor-data callbacks)
              command #(let [lang (:language (editor-data))
                             state (get-state lang)
                             lang-command (when state
                                            (get-in @state [:editor/commands cmd-key :command]))]
                         (if lang-command
                           (lang-command)
                           ((:notify callbacks) {:type :error
                                                 :title "Command not supported"
                                                 :message (str "Command " (name cmd-key)
                                                               " is not supported for " (name lang)
                                                               " language.")})))]]
    [cmd-key {:command command}]))

(defn disconnect! [lang]
  (set-commands lang [])
  (swap! connections update :states dissoc lang)
  #_
  (fn [connections]
    (let [without (dissoc connections lang)]
      (if (-> without count (<= 1))
        (do
          (some-> without :Config deref :config/watcher .close)
          (dissoc without :Config))
        without)))
  (when (-> @connections :states count (<= 1))
    (prn :done)
    (reset! connections empty-state)))

(prn :all? @connections)
