(ns corax.log
  (:require [clj-stacktrace.repl :refer [pst-str]]
            [clojure.pprint :refer [pprint]]))

(defprotocol Logger
  (ok [this sentry-id event]
    "Called when the event has been reported successfully to
    Sentry. `sentry-id` is the Sentry-assigned ID of the event.")
  (failure [this http-response event]
    "Called when the event reporting failed and Sentry returned an
    error. `http-response` is the non-200 HTTP response returned by
    Sentry.")
  (error [this exception event]
    "Called when the event reporting failed with an
    exception. `exception` is the exception that was thrown.")
  (invalid-dsn [this dsn event]
    "Called when the provided DSN could not be parsed correctly.")
  (non-serializable-event [this exception event]
    "Called when the provided event could not be serialized to
    JSON. `exception` is the exception that the serialization code
    threw."))

(defn- pprint-event [event]
  (println "Event: " (with-out-str (pprint event))))

(defrecord CoraxLogger []
  Logger
  (ok [this sentry-id event]
    (println "Corax - Sentry event ID:" sentry-id)
    (pprint-event event))

  (failure [this http-response event]
    (println "Corax - Sentry returned an error")
    (pprint http-response)
    (pprint-event event))

  (error [this exception event]
    (println "Corax - Unexpected exception when reporting event")
    (println (pst-str exception))
    (pprint-event event))

  (invalid-dsn [this dsn event]
    (println "Corax - DSN parsing failed")
    (println "DSN:" dsn)
    (pprint-event event))

  (non-serializable-event [this exception event]
    (println "Corax - Event cannot be serialized to JSON")
    (println (pst-str exception))
    (pprint-event event)))
