(ns budb.core
  (:require [konserve.memory :as mem]
            [konserve.core :as k]
            [budb.helpers :as h]
            [taoensso.timbre :as log :include-macros true]
    #?(:clj
            [superv.async :refer [<? >? go-loop-try go-try S] :as superv]
       :cljs
       [cljs.core.async :as async :refer [<! >! put!]])
    #?(:clj
            [clojure.core.async :as async :refer [<! >! go go-loop put!]]
       :cljs
       [superv.async :refer [<? >? go-loop-try go-try S] :as superv])
    #?(:clj
            [clojure.spec.alpha :as s]
       :cljs [cljs.spec :as s :include-macros true]))
  #?(:cljs
     (:require-macros
       [cljs.core.async.macros :refer [go go-loop]])))

;; Spec
;; ----

;; ### CRD(T/V)

(s/def :crdt/update
  (s/keys :req-un [:crdt/valid? :crdt/prepare
                   :crdt/appliable? :crdt/do]))

(s/def :crdt/query any?)

(s/def :crdt/updates
  (s/every-kv keyword?
              :crdt/update))

(s/def :crdt/queries
  (s/every-kv keyword?
              :crdt/query))

(s/def :crdt/state
  (s/keys :req-un [:crdt/valid? :crdt/initial]))

(s/def ::crdt
  (s/keys :req-un [:crdt/state :crdt/updates :crdt/queries]))

(s/def :crdv/replica-id number?)
(s/def ::crdv (s/keys :req-un [::crdt ::store :crdv/replica-id :crdv/state]))

(def crdt-spec ::crdt)
(def crdv-spec ::crdv)


;; ### Messages

(s/def :message/basic
  (s/keys :req-un [::kind ::feedback ::arg]
          :opt-un [::replayed?]))

(s/def :crdv/replica-version number?)

(s/def :message/versioned
  (s/merge :message/basic
           (s/keys :req-un [:crdv/replica-id :crdv/replica-version])))


;; Interface
;; ---------

(defprotocol CRDV
  (replica-id [this])
  (wal [this])
  (sync! [this store]))


;; API
;; ---

(defmacro assert-valid? [spec x]
  `(let [spec# ~spec
         x# ~x]
     (when-not (s/valid? spec# x#)
       (ex-info "Schema validation failed" (s/explain spec# x#)))))

(defn ->log [msg]
  (dissoc msg :feedback))

(defn state? [crdv]
  (assert-valid? ::crdv crdv)
  @(:state crdv))

(defn F [crdv msg f-name]
  (get-in crdv [:crdt :updates (:kind msg) f-name]))

(defn valid? [crdv {:keys [kind arg] :as msg}]
  ((F crdv msg :valid?) (state? crdv) arg))

(defn prepare! [{:keys [replica-id] :as crdv} version {:keys [kind arg] :as msg}]
  (let [new-arg ((F crdv msg :prepare) replica-id (state? crdv) version arg)]
    (assoc msg :arg new-arg :replica-id replica-id :replica-version version)))

(defn appliable? [{:keys [state] :as crdv} {:keys [kind arg] :as msg}]
  ((F crdv msg :appliable?) (state? crdv) arg))

(defn do! [{:keys [state] :as crdv} {:keys [kind arg] :as msg}]
  (swap! state (F crdv msg :do) arg))

(defn filter-for [kind]
  (fn [in]
    (let [c (async/chan 1 (comp (filter #(or (= (:ack/kind %) kind)
                                             (= (:status %) :fail)))
                                (map :status)))]
      (async/pipe in c)
      c)))
(defn until-ack [kind chan]
  ((filter-for kind) chan))

;; The CRDT relies on operations that are prepared locally then persisted and replicated
;; on other machines.
;; The operations is applied to the state (:do) to update it.
;; The property of the CRDT is that eventually every replica will show the same
;; state.
;; On the condition that:
;; - every update eventually arrive to every replicas
;; - the happened-before property is preserved: the set of operations {O_prev}
;;   that where applied on the current replica when it ran O_new is also
;;   applied on every replica before they apply O_new.
;;   (preserve the history/causality)

;; There's the CRDT definition {operations, queries}
;; it relies on the CRDT platform that deals with serializing inputs,
;; persisting and replicating the messages.

;; Each operation has a replica-id and a replica-version, this will be used
;; During replication.
;; Locally, each replica serialize the message in its WAL then update its state.
;; The replication works as follow:
;; - A replica pulls the "outstanding" operations from another replica
;; - Each operation is passed down the persist & update pipeline.

(defn msg->log [msg]
  (assert-valid? :message/versioned msg)
  (select-keys msg [:kind :arg :replica-id :replica-version]))

(defn make-persist [{:keys [store] :as crdv} in out]
  (assert-valid? ::crdv crdv)
  (go-loop [{:keys [feedback replica-id replica-version] :as msg} (<! in) known-versions {}]
    (log/info "PERSIST processing:" (->log msg))
    (try
      (if (<= (get known-versions replica-id -1) replica-version)
        (do
          (assert-valid? :message/versioned msg)
          (log/debug "PERSIST persists")
          (when-not (:replayed? msg)
            (<! (k/append store :wal (msg->log msg))))
          (log/debug "PERSIST checks applicability")
          (when-not (appliable? crdv msg)
            (throw (ex-info "Cannot apply the given message, something bad happend" msg)))
          (log/debug "PERSIST applies to the state")
          (do! crdv msg)
          (log/debug "PERSIST pass to the next")
          (do (>! feedback {:status :success :ack/kind :persisted})
              (>! out msg)))
        (do (>! feedback {:status :skipped :ack/kind :persisted})))
      (catch #?(:clj Exception :cljs js/Error) e
        (log/error "PERSIST failed with" e)
        (log/error e)
        (>! feedback {:status :fail :ack/kind :persisted :exception e})))
    (log/debug "PERSIST complete")
    (recur (<! in)
           (update known-versions replica-id (fn [x] (or x 0) replica-version)))))

;; The prepare phase validate the client's messages and apply the CRDT prepare function
;; - It sends a :status (:fail/:success) and a :ack/kind :prepared
;;   when the message is sent to the next phase
;; - It forward the message, with the crdt arg ready for persist & remote sync
(defn make-preparer [crdv in out]
  (assert-valid? ::crdv crdv)
  (go-loop [{:keys [kind arg feedback] :as msg} (<! in) version 0]
    (log/info "PREPARER processing:" (->log msg))
    (try
      (cond
        ;; A message that have already been prepared
        (s/valid? :message/versioned msg)
        (do (log/debug "VALIDATOR forwarding:" (->log msg))
            (>! out msg))
        ;; A fresh local message
        (s/valid? :message/basic msg)
        (if (valid? crdv msg)
          (let [prepared (prepare! crdv version msg)]
            (log/debug "VALIDATOR passing:" (->log prepared))
            (do (>! feedback {:status :success :ack/kind :prepared})
                (>! out prepared)))
          (throw (ex-info "Validation failed" {}))))
      (catch #?(:clj Exception :cljs js/Error) e
        (log/info "VALIDATOR failed with" e)
        (>! feedback {:status :fail :ack/kind :prepared :exception e})))
    (recur (<! in)
           (if (= (:replica-id msg) (:replica-id crdv))
             (max (:replica-version msg) version)
             (inc version)))))

(defn replica-id! [store & {:keys [default]}]
  (go (let [current (<! (k/get-in store [:replica-id]))]
        (log/info "Current replica id=" current)
        (if (nil? current)
          (let [x (default)]
            (<! (k/assoc-in store [:replica-id] x))
            x)
          current))))

(defn make! [crdt & {:keys [store replica-id]}]
  (assert (s/valid? crdt-spec crdt) (s/explain crdt-spec crdt))
  (go
    (let [store (if (= :mem store) (<! (mem/new-mem-store))
                                   store)
          replica-id (<! (replica-id! store :default
                                      #(if (nil? replica-id)
                                         (h/time-ordered-unique-id)
                                         replica-id)))
          state (atom (get-in crdt [:state :initial]))]
      (log/info "Made CRDT for replica-id=" replica-id)
      {:crdt crdt :store store :state state :replica-id replica-id})))

(defn log-stream [name store]
  (let [chan (async/chan)]
    (go (try
          (log/info "Pull wal from name=" name)
          (let [[append-log last-id first-id] (<? S (k/get-in store [name]))]
            (if (nil? first-id)
              (do (log/info "Empty log. Leaving")
                  (async/close! chan))
              (<? S (go-loop-try S [{:keys [next elem]} (<? S (k/get-in store [first-id]))]
                      (log/debug "Got ELEM=" elem)
                      (if (nil? elem)
                        (async/close! chan)
                        (do (>? S chan elem)
                            (recur (<? S (k/get-in store [next])))))))))
          (catch #?(:clj Throwable :cljs js/Error) e
            (log/error "Log Stream for name=" name "failed")
            (>! chan e)
            (async/close! chan))))
    chan))

(defn store-wal-stream [store]
  (log-stream :wal store))

(defn wal-stream [crdv]
  (store-wal-stream (:store crdv)))

(defn play! [wal-stream crdv & {:keys [replayed?] :or {replayed? false}}]
  (log/info "Replaying from wal stream to crdv" (:replica-id crdv))
  (go-loop [current (<! wal-stream)]
    (if (nil? current)
      :success
      (let [feedback (async/chan)]
        (log/debug "REPLAY pass:" current)
        (>! (-> crdv :chans :in) (assoc current :feedback feedback :replayed? replayed?))
        ;; TODO: ASSERT SUCCESS
        (<! (until-ack :persisted feedback))
        (recur (<! wal-stream))))))

(defn start! [crdv]
  (let [in (async/chan)
        to-persist (async/chan)
        persisted (async/chan)
        persisted-mult (async/mult persisted)]
    (assoc crdv
      :chans {:in        in :to-persist to-persist
              :persisted persisted :persisted-mult persisted-mult}
      :preparer (make-preparer crdv in to-persist)
      :persister (make-persist crdv to-persist persisted))))

(defn replay! [crdv]
  (go (<! (play! (wal-stream crdv) crdv :replayed? true))
      crdv))

(defn update! [{:keys [chans] :as crdv} update-id & args]
  (assert (s/valid? ::crdv crdv) (s/explain ::crdv crdv))
  (let [feedback (async/chan)]
    (go (>! (:in chans) {:kind update-id :feedback feedback :arg (if (= 1 (count args))
                                                                   (first args)
                                                                   args)}))
    feedback))

(defn query? [crdv query-id & args]
  (let [q (get-in crdv [:crdt :queries query-id])]
    (q (state? crdv))))

(defn remote-push [{:keys [store] :as crdv} remote-store]
  (let [local-replica-id (:replica-id crdv)
        wal (wal-stream crdv)]
    (log/info "Remote Sync from replica" (:replica-id crdv))
    (go-loop [{:keys [replica-id replica-version] :as msg} (<! wal)]
      (if (nil? msg)
        (do (log/info ":success, leaving")
            :success)
        (do
          (log/info "Checking syncability for:" msg "for replica:" replica-id)
          (when (and (= local-replica-id replica-id)
                     (> replica-version (or (<! (k/get-in store [:sync replica-id])) -1)))
            (log/info "Syncing message:" msg "for replica:" replica-id)
            (<! (k/append remote-store :wal msg))
            (<! (k/assoc-in (:store crdv) [:sync replica-id] replica-version)))
          (recur (<! wal)))))))

(defn remote-pull [crdv remote-store]
  (play! (store-wal-stream remote-store) crdv))

