(ns devbook.budb.register
  (:require
    [budb.test-env :refer [prefix]]
    [budb.cljs.test-helpers :refer [make-db]]
    [budb.core :as c]
    [budb.register :as r]
    [konserve.core :as k]
    [reagent.core :as reagent :refer [atom]]
    [cljs.pprint :refer [pprint]]

    [konserve.memory :refer [new-mem-store]]
    [konserve.indexeddb :refer [new-indexeddb-store]]
    [konserve-firebase.store :refer [new-firebasedb-store]]

    [cljs.core.async :refer [>! <!]]
    [taoensso.timbre :as log :include-macros true])
  (:require-macros
    [devcards.core :refer [defcard deftest defcard-rg]]
    [cljs.core.async.macros :refer [go]]))

(defonce mem-store (atom nil))
(defonce firebase-store (atom nil))
(defonce indexed-store (atom nil))

(go (reset! mem-store (<! (new-mem-store))))
(go (reset! firebase-store (<! (new-firebasedb-store
                                 (make-db)
                                 :prefix [prefix "devcards" "register-test"]))))
(go (reset! indexed-store (<! (new-indexeddb-store (str prefix "-devcards-register-test")))))

(defcard-rg local-register
  (let [r1 (atom nil)
        v1 (atom 0)
        r2 (atom nil)
        v2 (atom 0)]
    (go
      (reset! r1 (<! (r/make-lww nil :replica-id 1)))
      (reset! r2 (<! (r/make-lww nil :replica-id 2)))
      (<! (r/set-value! @r1 1 :persisted))
      (<! (r/set-value! @r2 1 :persisted))
      (reset! v1 1)
      (reset! v2 1))
    (fn []
      [:div {}
       [:h1 {} "Mem Store"]
       [:div.mem-store {}
        [:pre {:style {:overflow "auto" :max-height "420px"}}
         (when @mem-store
           (with-out-str (pprint @(:state @mem-store))))]]
       [:h1 {} "Registers"]
       [:button.btn {:className "btn-default"
                     :on-click  (fn [e]
                                  (.preventDefault e)
                                  (go (<! (c/sync! @r1 @mem-store))
                                      (<! (c/sync! @r2 @mem-store))
                                      (<! (c/sync! @r1 @mem-store))
                                      (reset! v1 (r/get-value @r1))
                                      (reset! v2 (r/get-value @r2))))}
        "Sync"]
       [:button.btn {:className "btn-default"
                     :on-click  (fn [e]
                                  (.preventDefault e)
                                  (go (<! (r/set-value! @r1
                                                        (inc (r/get-value @r1))
                                                        :persisted))
                                      (reset! v1 (r/get-value @r1))))}
        @v1
        "+1"]
       [:button.btn {:className "btn-default"
                     :on-click  (fn [e]
                                  (.preventDefault e)
                                  (go (<! (r/set-value! @r2
                                                        (inc (r/get-value @r2))
                                                        :persisted))
                                      (reset! v2 (r/get-value @r2))))}
        @v2
        "+1"]])))

(defcard-rg remote-register
  (let [r1 (atom nil)
        v1 (atom 0)
        r2 (atom nil)
        v2 (atom 0)
        remote-state (atom {})]
    (go
      (reset! r1 (<! (r/make-lww nil :replica-id 1)))
      (reset! r2 (<! (r/make-lww nil :replica-id 2)))
      (reset! v1 0)
      (reset! v2 0))
    (fn []
      [:div {}
       [:h1 {} "Firebase Store"]
       [:div.mem-store {}
        [:pre {:style {:overflow "auto" :max-height "420px"}}
         (when @firebase-store
           (with-out-str (pprint @remote-state)))]]
       [:h1 {} "Registers"]
       [:button.btn {:className "btn-default"
                     :on-click  (fn [e]
                                  (.preventDefault e)
                                  (go (<! (c/sync! @r1 @firebase-store))
                                      (<! (c/sync! @r2 @firebase-store))
                                      (<! (c/sync! @r1 @firebase-store))
                                      (reset! remote-state
                                              (<! (k/get-in @firebase-store [])))
                                      (reset! v1 (r/get-value @r1))
                                      (reset! v2 (r/get-value @r2))))}
        "Sync"]
       [:button.btn {:className "btn-default"
                     :on-click  (fn [e]
                                  (.preventDefault e)
                                  (go (<! (r/set-value! @r1
                                                        (inc (or (r/get-value @r1) 0))
                                                        :persisted))
                                      (reset! v1 (r/get-value @r1))))}
        @v1
        "+1"]
       [:button.btn {:className "btn-default"
                     :on-click  (fn [e]
                                  (.preventDefault e)
                                  (go (<! (r/set-value! @r2
                                                        (inc (or (r/get-value @r2) 0))
                                                        :persisted))
                                      (reset! v2 (r/get-value @r2))))}
        @v2
        "+1"]])))

(defcard-rg remote+local-register
  (let [r1 (atom nil)
        v1 (atom 0)
        r2 (atom nil)
        v2 (atom 0)
        remote-state (atom {})]
    (go
      (reset! r1 (<! (r/make-lww nil :replica-id 1)))
      (reset! r2 (<! (r/make-lww @indexed-store :replica-id 2)))
      (reset! v1 0)
      (reset! v2 0))
    (fn []
      [:div {}
       [:h1 {} "Firebase Store"]
       [:div.mem-store {}
        [:pre {:style {:overflow "auto" :max-height "420px"}}
         (when @firebase-store
           (with-out-str (pprint @remote-state)))]]
       [:h1 {} "Register"]
       [:button.btn {:className "btn-default"
                     :on-click  (fn [e]
                                  (.preventDefault e)
                                  (go (<! (c/sync! @r1 @firebase-store))
                                      (<! (c/sync! @r2 @firebase-store))
                                      (<! (c/sync! @r1 @firebase-store))
                                      (reset! remote-state
                                              (<! (k/get-in @firebase-store [])))
                                      (reset! v1 (r/get-value @r1))
                                      (reset! v2 (r/get-value @r2))))}
        "Sync"]
       [:button.btn {:className "btn-default"
                     :on-click  (fn [e]
                                  (.preventDefault e)
                                  (go (<! (r/set-value! @r1
                                                        (inc (or (r/get-value @r1) 0))
                                                        :persisted))
                                      (reset! v1 (r/get-value @r1))))}
        @v1
        "+1"]
       [:button.btn {:className "btn-default"
                     :on-click  (fn [e]
                                  (.preventDefault e)
                                  (go (<! (r/set-value! @r2
                                                        (inc (or (r/get-value @r2) 0))
                                                        :persisted))
                                      (reset! v2 (r/get-value @r2))))}
        @v2
        "+1 [indexeddb]"]])))

