(ns figwheel-testbook.reporting.render
  (:require
    [figwheel-testbook.reporting.report :refer [summary-0 summary-1 summary-2]]
    [reagent.core :as r]
    [clojure.string :as string]
    [cljs.pprint :as pprint]))

(defonce state (r/atom {:ns {} :test {}}))

(defn enumerate
  "For each x_n in XS, returns '([0, x_0], [1, x_1], ...)"
  [xs]
  (map vector (iterate inc 0) xs))

(defn color-favicon-data-url [color]
  (let [cvs (.createElement js/document "canvas")]
    (set! (.-width cvs) 16)
    (set! (.-height cvs) 16)
    (let [ctx (.getContext cvs "2d")]
      (set! (.-fillStyle ctx) color)
      (.fillRect ctx 0 0 16 16))
    (.toDataURL cvs)))

(defn change-favicon-to-color [color]
  (let [icon (.getElementById js/document "favicon")]
    (set! (.-href icon) (color-favicon-data-url color))))

(defn pretty [x]
  (with-out-str
    (pprint/pprint x)))

(defn show-error [{:keys [file line column type message context actual expected] :as err}]
  [:div.col-xs-12 {}
   ;[:div.row {}
   ; (pr-str err)
   [:div.row {}
    [:div.col-lg-12 {}
     [:p {} "Message"]
     [:pre {} message]]]
   [:div.row {}
    [:div.col-xs-2 {}
     (->> context reverse (string/join "-"))
     (str file " l." line)]
    [:div.col-xs-10 {}
     [:div.row {}
      [:p {} "Got"]
      [:pre {} (pretty actual)]]
     [:div.row {}
      [:p {} "Expected"]
      [:pre {} (pretty expected)]]]]])

(defn show-fail [{:keys [file line column type expected actual message context] :as err}]
  [show-error err])

(defn show-test [ns test-name {:keys [start end fails errors] :as content}]
  (let [summary (summary-2 content)]
    [:div.test {:className (str "row "
                                (case (:status summary)
                                  :success "bg-success"
                                  :running "bg-primary"
                                  "bg-warning"))}
     [:div.col-xs-1 {:className "text-center"}
      (case (:status summary)
        :running
        [:i {:className "fa fa-spinner fa-spin"}]
        :success
        [:i {:className "fa fa-check"}]
        :failed
        [:i {:className "fa fa-warning"}])]
     [:div.col-xs-8 {:className "text-center"}
      (str " " (name test-name))]
     [:div.col-xs-3 {:className "text-center"}
      (case (:status summary)
        :running "..."
        (str (/ (:duration summary) 1000.0) "s"))]
     (when fails
       (for [[i fail] (enumerate fails)]
         ^{:key i} [show-fail fail]))
     (when errors
       (for [[i error] (enumerate errors)]
         ^{:key i} [show-error error]))]))

(defn show-ns [ns content]
  (let [summary (summary-1 content)]
    [:div.ns {:className (str "row "
                              (case (:status summary)
                                :success "bg-success"
                                :running "bg-primary"
                                "bg-warning"))}
     [:div.col-xs-12
      [:h2 {:on-click
            (fn [e]
              (.preventDefault e)
              (swap! state #(update-in % [:ns ns :show?] not)))}
       (case (:status summary)
         :running [:i {:className "fa fa-spinner fa-spin"}]
         :success [:i {:className "fa fa-check"}]
         [:i {:className "fa fa-warning"}])
       (str " " (name ns))
       [:small {}
        (str (:running summary) "/" (:failed summary) "/" (:success summary))]]]
     (when (or (= :failed (:status summary))
               (get-in @state [:ns ns :show?] false))
       (for [[test-name content] content]
         ^{:key test-name} [show-test ns test-name content]))]))

(defn show-all [results]
  (let [summary (summary-0 results)]
    (case (:status summary)
      :running (change-favicon-to-color "#00D")
      :success (change-favicon-to-color "#0D0")
      :failed (change-favicon-to-color "#D00"))
    [:div.container {}
     [:div.row {:id "header"}
      [:div.col-xs-1 {:className "text-center"}
       [:h1 {} (case (:status summary)
                 :running [:i {:className "fa fa-spinner fa-spin"}]
                 :success [:i {:className "fa fa-check"}]
                 [:i {:className "fa fa-warning"}])]]
      [:div.col-xs-9 {:className "text-center"}
       [:h1 {} "Report"]]
      [:div.col-xs-2 {:className "text-center"}
       [:h1 {}
        (str (:running summary) "/" (:failed summary) "/" (:success summary))]]]
     (for [[ns-name content] results]
       ^{:key ns-name} [show-ns ns-name content])]))


