(ns clojush.util
  (:require [clojure.math.numeric-tower :as math]
            [clojure.zip :as zip]
            [clojure.walk :as walk])
  (:use [clojush.globals]
        [clojush.random]))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; utilities

(defn ensure-list [thing] ;; really make-list-if-not-seq, but close enough for here
  (if (seq? thing)
    thing
    (list thing)))

(defn print-return 
  "Prints the provided thing and returns it."
  [thing]
  (println thing)
  thing)

(defn keep-number-reasonable
  "Returns a version of n that obeys limit parameters."
  [n]
  (cond
    (integer? n)
    (cond
      (> n max-number-magnitude) max-number-magnitude
      (< n (- max-number-magnitude)) (- max-number-magnitude)
      :else n)
    :else
    (cond
      (> n max-number-magnitude) (* 1.0 max-number-magnitude)
      (< n (- max-number-magnitude)) (* 1.0 (- max-number-magnitude))
      (and (< n min-number-magnitude) (> n (- min-number-magnitude))) 0.0
      :else n)))

(defn count-points 
  "Returns the number of points in tree, where each atom and each pair of parentheses 
   counts as a point."
  [tree]
  (if (seq? tree)
    (+ 1 (apply + (map count-points tree)))
    1))

(defn code-at-point 
  "Returns a subtree of tree indexed by point-index in a depth first traversal."
  [tree point-index]
  (let [index (mod (math/abs point-index) (count-points tree))
        zipper (zip/seq-zip tree)]
    (loop [z zipper i index]
      (if (zero? i)
        (zip/node z)
        (recur (zip/next z) (dec i))))))

(defn insert-code-at-point 
  "Returns a copy of tree with the subtree formerly indexed by
   point-index (in a depth-first traversal) replaced by new-subtree."
  [tree point-index new-subtree]
  (let [index (mod (math/abs point-index) (count-points tree))
        zipper (zip/seq-zip tree)]
    (loop [z zipper i index]
      (if (zero? i)
        (zip/root (zip/replace z new-subtree))
        (recur (zip/next z) (dec i))))))

(defn remove-code-at-point 
  "Returns a copy of tree with the subtree formerly indexed by
   point-index (in a depth-first traversal) removed. If removal would
   result in an empty list then it is not performed. (NOTE: this is different
   from the behavior in other implementations of Push.)"
  [tree point-index]
  (let [index (mod (math/abs point-index) (count-points tree))
        zipper (zip/seq-zip tree)]
    (if (zero? index)
      tree ;; can't remove entire tree
      (loop [z zipper i index]
        (if (zero? i)
          (zip/root (zip/remove z))
          (if (and (= i 1) ;; can't remove only item from list
                   (seq? (zip/node z))
                   (= 1 (count (zip/node z))))
            (zip/root z) ;(zip/remove z))
            (recur (zip/next z) (dec i))))))))

(defn remove-parens-at-point
  "Returns a copy of tree with the parens of the subtree formerly indexed by
   point-index (in a depth-first traversal) removed. This
   only works if that subtree is a sequence, otherwise this
   just returns the tree. Also, can't remove from index 0,
   since you can't remove the outmost parens."
  [tree point-index]
  (let [index (mod (math/abs point-index) (count-points tree))
        zipper (zip/seq-zip tree)]
    (if (zero? index)
      tree ;; can't remove entire tree's parens
      (let [z-found (loop [z zipper i index]
                      (if (zero? i)
                        z
                        (recur (zip/next z) (dec i))))]
        (if (not (seq? (zip/node z-found))) ;can't un-paren non-seq things
          (zip/root z-found)
          (if (empty? (zip/children z-found)) ;seq is empty, just remove it
            (zip/root (zip/remove z-found))
            (let [children (zip/children z-found)]
              (loop [ch (butlast children)
                     z (zip/replace z-found (last children))]
                (if (empty? ch)
                  (zip/root z)
                  (recur (rest ch)
                         (zip/insert-left z (first ch))))))))))))

(defn truncate
  "Returns a truncated integer version of n."
  [n]
  (if (< n 0)
    (math/round (math/ceil n))
    (math/round (math/floor n))))

(defn walklist
  "Like walk, but only for lists."
  [inner outer form]
  (cond
    (list? form) (outer (apply list (map inner form)))
    (seq? form) (outer (doall (map inner form)))
    :else (outer form)))

(defn postwalklist
  "Like postwalk, but only for lists"
  [f form]
  (walklist (partial postwalklist f) f form))

(defn postwalklist-replace
  "Like postwalk-replace, but only for lists."
  [smap form]
  (postwalklist (fn [x] (if (contains? smap x) (smap x) x)) form))

(defn subst
  "Returns the given list but with all instances of that (at any depth)                                   
   replaced with this. Read as 'subst this for that in list'. "
  [this that lst]
  (postwalklist-replace {that this} lst))

(defn contains-subtree 
  "Returns true if tree contains subtree at any level. Inefficient but
   functional implementation."
  [tree subtree]
  (or 
    (= tree subtree)
    (not (= tree (subst (gensym) subtree tree)))))

(defn containing-subtree
  "If tree contains subtree at any level then this returns the smallest
   subtree of tree that contains but is not equal to the first instance of
   subtree. For example, (contining-subtree '(b (c (a)) (d (a))) '(a)) => (c (a)).
   Returns nil if tree does not contain subtree."
  [tree subtree]
  (cond 
    (not (seq? tree)) nil
    (empty? tree) nil
    (some #{subtree} tree) tree
    :else (some (fn [smaller-tree]
                  (containing-subtree smaller-tree subtree))
                tree)))

(defn all-items
  "Returns a list of all of the items in lst, where sublists and atoms all
   count as items. Will contain duplicates if there are duplicates in lst.
   Recursion in implementation could be improved."
  [lst]
  (cons lst (if (seq? lst)
              (apply concat (doall (map all-items lst)))
              ())))

(defn not-lazy
  "Returns lst if it is not a list, or a non-lazy version of lst if it is."
  [lst]
  (if (seq? lst)
    (apply list lst)
    lst))

;; backtrace abbreviation, to ease debugging
(defn bt []
  (.printStackTrace *e))

(defn insert-randomly
  "Returns lst with thing inserted in a random location. If lst is not a list then
it will first be wrapped in a list."
  [thing lst]
  (let [tree (ensure-list lst)
        loc (inc (lrand-int (dec (count-points tree))))]
    (zip/root
      ((lrand-nth [zip/insert-left zip/insert-right])
        (loop [z (zip/seq-zip tree) i 0]
          (if (= i loc)
            z
            (recur (zip/next z) (inc i))))
        thing))))
                             