(ns timewords.standard.joda-formats
  (:require [clojure.string :as str])
  (:import (java.util Locale TimeZone Date)
           (org.joda.time.format ISODateTimeFormat DateTimeFormat DateTimeFormatter)
           (org.joda.time DateTime LocalDateTime)
           (org.joda.time.chrono LenientChronology ISOChronology)))

(def lenient-chronology (LenientChronology/getInstance (ISOChronology/getInstance)))

(defn fmt [pattern & [locale default-year lenient]]
  (cond-> (DateTimeFormat/forPattern pattern)
          locale (.withLocale locale)
          default-year (.withDefaultYear default-year)
          lenient (.withChronology lenient-chronology)))

(defn common-formatters [locale]
  [(ISODateTimeFormat/dateTimeParser)
   (fmt "yyyy-MM-dd HH:mm" locale)
   (fmt "yyyy-MM-dd HH:mm:ss" locale)
   (fmt "yyyy-MM-dd, HH:mm" locale)
   (fmt "dd/MM/yy HH:mm" locale)
   (fmt "yyyy/MM/dd" locale)
   (fmt "dd/MM/yyyy, HH:mm" locale)
   (fmt "dd/MM/yyyy" locale)
   (fmt "MM/dd/yyyy" locale)
   (fmt "MMMM yyyy" locale)
   (fmt "dd'st' MMMM yyyy" locale)
   (fmt "dd'nd' MMMM yyyy" locale)
   (fmt "dd'rd' MMMM yyyy" locale)
   (fmt "dd'th' MMMM yyyy" locale)
   (fmt "EEE, dd'st' MMMM yyyy" locale)
   (fmt "EEE, dd'nd' MMMM yyyy" locale)
   (fmt "EEE, dd'rd' MMMM yyyy" locale)
   (fmt "EEE, dd'th' MMMM yyyy" locale)
   (fmt "EEE MMMM dd, yyyy" locale)
   (fmt "EEE MMMM dd, yyyy h:mma" locale)
   (fmt "EEE MMMM dd, yyyy h:mma z" locale)
   (fmt "EEE MMMM dd yyyy HH:mm 'UTC'Z" locale)
   (fmt "dd MMM yyyy HH:mm" locale)
   (fmt "MMM dd, yyyy HH:mm z" locale)
   (fmt "dd MMM yyyy" locale)
   (fmt "MMMM dd, yyyy h:mma" locale)
   (fmt "EEE, MMMM dd, yyyy" locale)
   (fmt "yyyy-MM-dd'T'HH:mm:ssZ" locale)
   (fmt "MMMM dd, yyyy h:mm a" locale)
   (fmt "MMMM dd, yyyy" locale)
   (fmt "MMMM. dd, yyyy" locale)
   (fmt "EEE MMMM dd HH:mm:ss z yyyy" locale)
   (fmt "yyyy-MM-dd HH:mm:ss z" locale)
   (fmt "EEE dd MMMM yyyy HH:mm" locale)
   (fmt "yyyy/MM/dd HH:mm:ss" locale)
   (fmt "yyyy-MM-dd HH:mm:ss.S" locale)
   (fmt "MM/dd/yyyy HH:mm" locale)
   (fmt "MM/dd/yyyy HH:mm:ss a z" locale)
   (fmt "MMMM dd, yyyy, h:mm a" locale)
   (fmt "MMMM dd, yyyy - h:mma" locale)
   (fmt "EEE, MMMM dd, yyyy, h:mm a" locale)
   (fmt "EEEE dd MMMM yyyy HH.mm z" locale)
   (fmt "yyyy-MM-dd '/' HH:mm" locale)
   (fmt "yyyy.MM.dd HH:mm" locale)
   (fmt "HH:mm yyyy.MM.dd" locale)
   (fmt "EEEE, dd. MMMM yyyy, HH:mm 'Uhr'" locale)
   (fmt "dd. MMMM yyyy, HH:mm 'Uhr'" locale)
   (fmt "EEEE, dd. MMMM yyyy" locale)
   (fmt "dd. MMMM yyyy" locale)
   (fmt "EEEE, dd.MM.yyyy, HH:mm" locale)
   (fmt "EEEE, dd.MM.yyyy" locale)
   (fmt "dd.MM.yyyy" locale)
   (fmt "MMMM dd" locale (.getYear (DateTime.)))
   (fmt "MMMM dd 'd.' HH:mm" locale (.getYear (DateTime.)))
   (fmt "yyyy/MM/dd HH:mm" locale)
   (fmt "yyyy MM dd HH:mm" locale)
   (fmt "yyyy MMMM dd" locale)
   (fmt "yyyy 'metų' MMMM dd" locale)
   (fmt "yyyy MMMM dd HH:mm" locale)
   (fmt "yyyy MMMM dd HH:mm:ss" locale)
   (fmt "yyyy 'm.' MMMM dd 'd.' HH:mm" locale)
   (fmt "yyyy MMMM dd'd.' HH:mm" locale)
   (fmt "yyyy-MM-dd '/' HH:mm" locale)
   (fmt "yyyy.MM.dd HH:mm" locale)
   (fmt "HH:mm yyyy.MM.dd" locale)
   (fmt "dd.MM.yyyy - HH:mm'h'" locale)
   (fmt "dd.MM.yyyy – HH:mm 'H.'" locale)
   (fmt "yyyy-MM-dd HH:mm:ss 'H'" locale)
   (fmt "dd/MM/yyyy HH:mm 'h'" locale)
   (fmt "dd/MM/yyyy - HH:mm" locale)
   (fmt "dd MMMM yyyy - HH:mm 'CEST'" locale)
   (fmt "dd MMMM yyyy - HH:mm" locale)
   (fmt "dd MMMM yyyy HH:mm'h' 'CEST'" locale)
   (fmt "dd MMMM'.' yyyy HH:mm" locale)
   (fmt "dd/MM/yyyy HH:mm'h'" locale)
   (fmt "dd/MM/yyyy HH:mm:ss 'CET'" locale)
   (fmt "dd/MM/yyyy HH:mm:ss" locale)
   (fmt "HH:mm - dd/MM/yy" locale)
   (fmt "EEEE, dd MMMM yyyy, HH:mm" locale)
   (fmt "EEEE, dd/MM/yyyy" locale)
   (fmt "EEEE dd MMMM yyyy" locale)
   (fmt "dd 'de' MMMM 'de' yyyy'.' HH:mm'h'" locale)
   (fmt "dd 'de' MMMM 'de' yyyy" locale)])

(defn normalize [text]
  (-> text
      (str/lower-case)
      (str/replace "kov " "kovo ")
      (str/replace "vasario" "vasaris")
      (str/replace "balandžio" "balandis")
      (str/replace "gegužės" "gegužė")
      (str/replace "liepos" "liepa")
      (str/replace "rugpjūčio" "rugpjūtis")
      (str/replace "rugsėjo" "rugsėjis")
      (str/replace "spalio" "spalis")
      (str/replace "lapkričio" "lapkritis")
      (str/replace "gruodžio" "gruodis")
      (str/replace "pst" "PST")
      (str/replace "edt" "EDT")
      (str/replace "bst" "GMT")
      (str/replace " gmt" " GMT")
      (str/replace "p.m." "pm")
      (str/replace "utc" "UTC")
      (str/replace "est" "EST")
      (str/replace "mar." "marzo")
      (str/replace " mar " " marzo ")))

(def used-locales (atom {}))

(defn get-formatters [locale]
  (if-let [used-formatters (get @used-locales locale)]
    used-formatters
    (let [new-formatters (common-formatters locale)]
      (swap! used-locales assoc locale new-formatters)
      new-formatters)))

(defn parse
  ([^String text]
   (parse text Locale/ENGLISH nil))
  ([^String text ^Locale locale]
   (parse text locale nil))
  ([^String text ^Locale locale ^DateTime document-time]
   (let [text (normalize text)
         formatters (get-formatters locale)]
     (for [^DateTimeFormatter formatter formatters
           :let [parsed (try
                          (let [^LocalDateTime pdate (.parseLocalDateTime formatter text)]
                            (.toDate (if (and (not (nil? document-time))
                                              (not= (.getYear document-time) (.getYear (DateTime.))))
                                       (.minusYears pdate (- (.getYear pdate) (.getYear document-time)))
                                       pdate)
                                     (TimeZone/getTimeZone "GMT")))
                          (catch Exception _ nil))]
           :when parsed] parsed))))
