(ns {{name}}.core
  (:require [reagent.core :as r]
            [reagent.session :as session]
            [secretary.core :as secretary :include-macros true]
            [goog.events :as events]
            [goog.history.EventType :as HistoryEventType]
            [markdown.core :refer [md->html]]
            [ajax.core :refer [GET POST]]
            [{{name}}.ajax :refer [load-interceptors!]]
            [{{name}}.websockets :as ws])
  (:import goog.History))



;; websocket stuff

(defonce messages (r/atom []))

(defn message-list []
  [:ul
   (for [[i message] (map-indexed vector @messages)]
     ^{:key i}
     [:li message])])

(defn valid-message?
  ""
  [message]
  (and (re-find #"[A-Z]" message)
       (not (empty? message))))

(defn check-input
  "checks input value against non-empty and a validation predicate
   and returns appropriate message"
  [val pred]
  (if (or (nil? val)
          (= "" val))
    "field cannot be empty"
    (if (pred val)
      "field is valid"
      "field is not valid")))

(defn input-elem
  "an input element template"
  [id value in-focus]
  [:input {:id id
           :name id
           :class "input-elem"
           :type id
           :required ""
           :value @value
           :on-change #(reset! value (-> % .-target .-value))
           :on-focus #(swap! in-focus not)
           :on-blur #(swap! in-focus not)}])

(defn input-with-msg
  "wraps and input element with a label and validation message"
  [label-val name elem-arg elem-valid req]
  (let [input-focus (r/atom false)] 
    (fn []
      [:div
       [:label label-val]
       [input-elem name elem-arg input-focus]
       [:div (check-input @elem-arg elem-valid)]])))

(defn message-elem
  ""
  [message-atom]
  (input-with-msg "Message:"
                  "message"
                  message-atom
                  valid-message?
                  true))

(defn send-message
  "a pseudo-form for shouting a message"
  []
  (let [message (r/atom "")]
    (fn []
      [:div {:class "ws-message-wrapper"}
       [:h3 "Shout a Message"]
       [message-elem message]
       [:button {:id "ws-shout-message"
                 :disabled (not (valid-message? @message))
                 :on-click #(do (ws/send-transit-msg! {:message @message})
                                (reset! message ""))}
        "shout"]])))

(defn update-messages! [{:keys [message]}]
  (swap! messages #(vec (take 10 (conj % message)))))

;; end websocket stuff


(defn about-page []
  [:div.container
   [:p "This is the story of {{name}}... work in progress"]])

(defn home-page
  "gol DOM layout"
  []
  (fn []
    [:div
     [:h1 "Shouting"]
     [:hr]
     [send-message]
     [message-list]]))

(def pages
  {:home #'home-page
   :about #'about-page})

(defn page []
  [(pages (session/get :page))])

;; -------------------------
;; Routes
(secretary/set-config! :prefix "#")

(secretary/defroute "/" []
  (session/put! :page :home))

(secretary/defroute "/about" []
  (session/put! :page :about))

;; -------------------------
;; History
;; must be called after routes have been defined
(defn hook-browser-navigation! []
  (doto (History.)
        (events/listen
          HistoryEventType/NAVIGATE
          (fn [event]
              (secretary/dispatch! (.-token event))))
        (.setEnabled true)))

;; -------------------------
;; Initialize app
(defn fetch-docs! []
  (GET (str js/context "/docs") {:handler #(session/put! :docs %)}))

(defn mount-components []
  ;(r/render [#'navbar] (.getElementById js/document "navbar"))
  (r/render [#'page] (.getElementById js/document "app")))

(defn init! []
  (load-interceptors!)
  (fetch-docs!)
  (hook-browser-navigation!)
  (ws/make-websocket! (str "ws://" (.-host js/location) "/ws") update-messages!)
  (mount-components))
