(ns janus.database
  (:require [clojure.java.jdbc :as sql]
            [clojure.java.io :as io]
            [clj-time.core :as t-c]
            [janus.dates])
  (:import [java.util Date]
           [org.postgresql.util PGInterval]
           [java.security MessageDigest]
           [java.math BigInteger]))

(defn- migrations-table-ddl
  "Migrations that have been applied are stored in the database so we
  know which ones have been run already. We also keep track of when it
  was run, and how long it took. We also store a hash generated from
  the source migration file."
  [table-name]
  (sql/create-table-ddl (str "IF NOT EXISTS " table-name)
                        [[:id "SERIAL PRIMARY KEY"]
                         [:filename "CHARACTER VARYING(255) NOT NULL UNIQUE"]
                         [:hash "CHARACTER(32) NOT NULL UNIQUE"]
                         [:execution-date "TIMESTAMP WITH TIME ZONE NOT NULL"]
                         [:duration "INTERVAL NOT NULL"]]
                        {:entities #(clojure.string/replace % "-" "_")}))

(defn ensure-migrations-table
  "Make sure that the migrations table exists. Will skip if it already
  does, so is safe to run anytime."
  [db-spec migrations-table]
  (sql/execute! db-spec [(migrations-table-ddl migrations-table)]))

(defn add-migration
  "Add the migration information into the table. We track the duration
  of migrations, so it expects the start time of the migration (in
  epoch ms)."
  [db-spec migrations-table migration]
  (let [migration (select-keys migration [:filename :hash :execution-date :duration])]
    (sql/insert! db-spec migrations-table
                 migration
                {:entities #(clojure.string/replace % "-" "_")})))

(defn get-applied-migrations
  "Query the database and get the list of migrations already applied."
  [db-spec]
  (sql/query db-spec ["SELECT filename, hash FROM migrations"]))

(defn sql-file-name
  "Given a file, returns the name of the file if it matches the
  expected pattern for SQL migrations."
  [file]
  (last (re-matches #"(.*?)(\d+ - .*\.sql)" (str file))))

(defn migrations-to-verify
  "Given a sequence of migration hashmaps and sequence of migrations
  already in the database, return the hash of any common migrations
  that we're to check that the filename matches."
  [existing-migrations new-migrations]
  (let [existing-migrations (set (map :hash existing-migrations))
        new-migrations (set (map :hash new-migrations))]
    (clojure.set/intersection existing-migrations new-migrations)))

(defn migrations-to-add
  "Find any migrations that are not in the database and return a set
  of the hash."
  [existing-migrations new-migrations]
  (let [existing-migrations (set (map :hash existing-migrations))
        new-migrations (set (map :hash new-migrations))]
    (clojure.set/difference  new-migrations existing-migrations)))

(defn timing-information
  "Given a migration hash-map, add timing information to it."
  [migration start-time]
  (merge migration
         {:execution-date start-time
          :duration (t-c/interval start-time
                                   (t-c/now))}))

(defn execute-migration
  "Given a hashmap that represents the migration, execute it and add
  it to the migrations table."
  [db-spec migration]
  (let [start-time (t-c/now)]
    (println (:filename migration))
    (sql/with-db-transaction [conn db-spec]
      (sql/execute! conn [(:sql-string migration)])
      (add-migration conn "migrations"
                     (timing-information migration start-time)))))

(defn process-migrations
  "Run the given migrations against the given database."
  [db-spec migrations]
  (let [migrations-to-do (migrations-to-add (get-applied-migrations db-spec)
                                            migrations)]
    (doall
     (map #(execute-migration db-spec %)
          (filter #(migrations-to-do (:hash %)) migrations)))))

(defn read-sql
  ""
  [file]
  (clojure.string/split (slurp file) #"(?m)\n\s*--;;\s*\n"))

(defn hash-string
  "Given a string, return the MD5 hash."
  [in-string]
  (let [algorithm (MessageDigest/getInstance "MD5")
        raw (.digest algorithm (.getBytes in-string))]
    (format "%032x" (BigInteger. 1 raw))))

(defn load-sql
  "Given a filename, load and return a hash-map of the migration."
  [file]
  (if-let [filename (sql-file-name file)]
    (let [sql-string (first (read-sql file))]
      {:filename filename
       :hash (hash-string sql-string)
       :sql-string sql-string})))

(defn load-directory
  "Given a directory path, return the migrations found there."
  [path]
  (map load-sql (rest (file-seq (io/file path)))))

(defn migrate!
  "Given a directory path and a database spec, run the migrations in
  that directory against the database."
  [db-spec path]
  (ensure-migrations-table db-spec "migrations")
  (process-migrations db-spec (sort-by :filename (load-directory path))))
