(ns macchiato.middleware
  (:require
    [clojure.set :refer [difference]]))

(defn update-middleware-meta [handler handler-middleware middleware-meta]
  (with-meta
    handler
    {:macchiato/middleware
     (conj handler-middleware middleware-meta)}))

(defn validate
  "middleware metadata can contain the following keys
  :id - id of the middleware function
  :required - a collection of the ids of middleware functions it requires to be present"
  [handler-middleware
   {:keys [id required] :as middleware-meta}]
  (when (not-empty (difference (set required) (set (map :id handler-middleware))))
    (throw (js/Error. (str id " is missing required middleware: " required))))
  middleware-meta)

(defn- middleware-from-handler [handler]
  (->> handler meta :macchiato/middleware (remove nil?) vec))

(defn validate-handler [handler]
  (let [middleware (middleware-from-handler handler)]
    (loop [[middleware-meta & handler-middleware] middleware]
      (when middleware-meta
        (validate handler-middleware middleware-meta)
        (recur handler-middleware)))
    handler))

(defn loaded? [middleware {:keys [id]}]
  (some #{id} (map :id middleware)))

(defn wrap
  ([handler middleware-fn]
   (wrap handler middleware-fn nil))
  ([handler middleware-fn opts]
   (let [handler-middleware (middleware-from-handler handler)
         middleware-meta    (-> middleware-fn meta :macchiato/middleware)]
     (if (loaded? handler-middleware middleware-meta)
       handler
       (update-middleware-meta
         (if opts
           (middleware-fn handler opts)
           (middleware-fn handler))
         handler-middleware
         middleware-meta)))))

(defn- remove-duplicate-middleware [handler-middleware middleware]
  (remove
    #(loaded?
       handler-middleware
       (-> (if (coll? %) (first %) %) meta :macchiato/middleware))
    middleware))

(defn wrap-middleware [handler & middleware]
  (let [handler-middleware  (middleware-from-handler handler)
        filtered-middleware (remove-duplicate-middleware handler-middleware middleware)]
    (-> (reduce
          (fn [handler middleware-fn]
            (let [[middleware-fn opts] (if (coll? middleware-fn) middleware-fn [middleware-fn])]
              (wrap handler middleware-fn opts)))
          (with-meta handler {:macchiato/middleware []})
          filtered-middleware)
        (vary-meta update :macchiato/middleware #(into % handler-middleware))
        (validate-handler))))

