(ns metaheuristics.annealing)

(defprotocol AnnealingProblem
  "A protocol that specifies the inputs to the annealing
  function."
  (temperature [this iteration] "Returns the temperature that we reach for the input iteration. Iteration increases by one
  for each 'time unit'.")
  (successor [this current-solution] "Returns a valid successor solution that can be reached from the current solution")
  (cost [this solution] "Returns the cost of the provided solution."))

(defn- shake
  "Helper function which simulates 'shaking' in simulated annealing."
  [current-solution candidate delta temperature]
  (let [threshold (Math/pow (Math/E) (float (/ delta temperature)))]
    (if (> (rand) threshold)
      current-solution
      candidate)))

(defn anneal
  "Attempts to find an optimal solution to the provided AnnealingProblem."
  ([problem initial-state] (anneal problem initial-state 0))
  ([problem current-solution iteration]
   (let [current-temp (temperature problem iteration)]
     (if (<= current-temp 0)
       current-solution
       ; We want to use `recur`, so no if-let here.
       (let [next-candidate (successor problem current-solution)
             delta (- (cost problem next-candidate) (cost problem current-solution))
             next-solution (if (pos? delta)
                             next-candidate
                             (shake current-solution next-candidate delta current-temp))]
         (recur problem next-solution (inc iteration)))))))